#include "iostm8s103f3.h"

//memory boundaries
#define  RAM_START            0x0000
#define  RAM_END              0x03FF
#define  OPTION_START         0x4800
#define  OPTION_END           0x480A
//#define  UBC_OPTION_LOCATION  0x4801
//#define  FLASH_START          0x8000
//#define  FLASH_END            0x9FFF
#define  FLASH_START          0x8000
#define  FLASH_END            0x97FF
//#define  FLASH_CLEAR_BYTE     0xA5
#define  EEPROM_START         0x4000
#define  EEPROM_END           0x427F
//#define  EEPROM_END           0x407F

#define BOOTV_START 0x8000
#define BOOTV_END   0x8003

#define RESET_VECTOR_ADDR	0x9EFC
#define RESET_VECTOR_ASM	"$9EFC"
/*void HSE_init()       
{
  //    ( )
  CLK_ECKR_bit.HSEEN  = 1;      //  HSE
  while(!CLK_ECKR_bit.HSERDY);  //    
  CLK_SWCR_bit.SWEN   = 1;      //     
  CLK_SWR     = 0xB4;           //  HSE   
  CLK_CKDIVR  = 0;              //   
  while(!CLK_SWCR_bit.SWIF);    //   
  CLK_SWCR_bit.SWIF   = 0;
}*/

void UART_init()
{
  CLK_PCKENR1 |= 1<<3;  //     UART  stm8s103f3
  
  PC_DDR_bit.DDR4 = 1; // TX
  PC_CR1_bit.C15 = 1;  //   Push-pull
  PC_CR2_bit.C25 = 0;  //   -  2 .
  
  PC_DDR_bit.DDR6 = 0; // RX
  PC_CR1_bit.C16 = 1;  //   Push-up
  PC_CR2_bit.C26 = 0;  //  
  
  UART1_CR1_bit.M     = 0;  //    8 
  UART1_CR1_bit.PCEN  = 0;  //    (   - )
  UART1_CR3_bit.STOP  = 0;  // 1  
  UART1_BRR2  = 0x03;       // : BRR2   BRR1
  UART1_BRR1  = 0x69;       // 9600 kbps  16 MHz
  UART1_CR2_bit.TEN   = 1;  //  
  UART1_CR2_bit.REN   = 1;  //  
}

void putc(char c)
{
  while(!(UART1_SR & MASK_UART1_SR_TXE));
  UART1_DR = c;
}

void puts(const char* str)
{
  while(*str)
  {
    if(*str == '\n') putc('\r');
    putc(*str);
    ++str;
  }
}

//     
char UART_getc()
{
  while(!(UART1_SR & MASK_UART1_SR_RXNE));  // UART1 mask have wrong bit description
  return UART1_DR;
}

#define XON   0x11
#define XOFF  0x13

static char XonXoff = 0;

#define GET_SAVEBUF_SIZE  32
#define GET_SAVETIMEOUT   30000

static char save_buffer[GET_SAVEBUF_SIZE];
static int save_count = 1;
static int save_total = 1;

void ungetc(char c)
{
  if(save_count)
  {
    //     
    save_buffer[--save_count] = c;
  }
}

char getc()
{
  if(save_count < save_total)
  {
    //    ,  
    return save_buffer[save_count++];
  }
  else
  {
    //    
    if(XonXoff)
	{
      //  ,   
      XonXoff = 0;
      putc(XON);
      save_total = save_count = 1;
	}
	//   
    return UART_getc();
  }
}

void saveInput()
{
  int delay;
  if(!XonXoff)
  {
    //  ,    
    XonXoff = 1;
    putc(XOFF);
  }
  //   ,      
  while(save_total < GET_SAVEBUF_SIZE) {
    for(delay = GET_SAVETIMEOUT; delay; --delay)
    {
	  if(UART1_SR & MASK_UART1_SR_RXNE)
	  {
        //    ,    
        save_buffer[save_total++] = UART1_DR;
		break;
      }
    }
	//    
    if(!delay) break;
  }
}

#define BACKSPACE 8
#define GETLINE_BUF_SIZE  80

char* gets()	//  .  
{
  static char buffer[GETLINE_BUF_SIZE];
  int count = 0;
  char c;
  while(count < GETLINE_BUF_SIZE - 1)
  {
    c = getc();
    if(c == '\r' || c == '\n')
    {
      puts("\n");
      break;
    }
    else if(c == BACKSPACE)
    {
      putc(c);
      count--;
    }
    else
    {
      putc(c);	// echo char
      buffer[count] = c;
      count++;
    }
  }
  //  
  buffer[count] = 0;
  saveInput();
  return &buffer[0];
}

const char* hexsyms = "0123456789ABCDEF";
void putHex(char data)
{
  putc(hexsyms[(data>>4)&0x0F]);
  putc(hexsyms[(data)&0x0F]);
}
void putVar(int var)
{
  putHex(var>>8);
  putHex(var&0xFF);
}

void unlock_PROG(void)
{
  //Unlock PROG memory
  FLASH_PUKR = 0x56;
  FLASH_PUKR = 0xAE;
}
	
void unlock_DATA(void)
{
 //Unlock DATA memory
  FLASH_DUKR = 0xAE; // Warning: keys are reversed on data memory !!!
  FLASH_DUKR = 0x56;
}

void lock_memory(void){
  // Lock program memory
  //FLASH_IAPSR = ~0x02;
  FLASH_IAPSR_bit.PUL = 0;
  // Lock data memory
  //FLASH_IAPSR = ~0x08;
  FLASH_IAPSR_bit.DUL = 0;
}

/*u8 WriteBufferFlash(u8 FAR* DataAddress, u8 DataCount, FLASH_MemType_TypeDef MemType){
  u32 Address = (u32) DataAddress;
  u8 * DataPointer = DataBuffer;
  u32 Offset;
  //set offset according memory type
  if(MemType == FLASH_MEMTYPE_PROG)
    Offset = FLASH_START;
  else
    Offset = EEPROM_START;
  //program beginning bytes before words
  while((Address % 4) && (DataCount))
  {
    *((PointerAttr u8*) Address) = *DataPointer;
		while( (FLASH->IAPSR & (FLASH_IAPSR_EOP | FLASH_IAPSR_WR_PG_DIS)) == 0);
		Address++;
    DataPointer++;
    DataCount--;
	}
  //program beginning words before blocks
	while(DataCount >= 4)
  {
		FLASH->CR2 |= (u8)0x40;
		FLASH->NCR2 &= (u8)~0x40;
		*((PointerAttr u8*)Address) = (u8)*DataPointer  ; 	 // Write one byte - from lowest address
		*((PointerAttr u8*)(Address + 1)) = *(DataPointer + 1); // Write one byte
		*((PointerAttr u8*)(Address + 2)) = *(DataPointer + 2); // Write one byte*
		*((PointerAttr u8*)(Address + 3)) = *(DataPointer + 3); // Write one byte - from higher address
    while( (FLASH->IAPSR & (FLASH_IAPSR_EOP | FLASH_IAPSR_WR_PG_DIS)) == 0);
		Address    += 4;
    DataPointer+= 4;
    DataCount  -= 4;
  }
  
  //program remaining bytes (after words)
  while(DataCount)
  {
    *((PointerAttr u8*) Address) = *DataPointer;
    while( (FLASH->IAPSR & (FLASH_IAPSR_EOP | FLASH_IAPSR_WR_PG_DIS)) == 0);
		Address++;
    DataPointer++;
    DataCount--;
  }
  
  return 1;
}//WriteBufferFlash*/

int WriteMemory(unsigned int base, const unsigned char* data, int len)
{
  //program remaining bytes (after words)
  while(len)
  {
    *((volatile unsigned char*) base) = *data;
    while( (FLASH_IAPSR & (MASK_FLASH_IAPSR_EOP | MASK_FLASH_IAPSR_WR_PG_DIS)) == 0);
    base++;
    data++;
    len--;
  }
  return len;
}

/*int ReadMemory(const char* base, char* data, int len)
{
  //program remaining bytes (after words)
  while(len)
  {
    *data = *((volatile char*) base);
    base++;
    data++;
    len--;
  }
  return len;
}*/

int hex2int(char c)
{
  if(c >= '0' && c <= '9') return c - '0';
  else if(c >= 'A' && c <= 'F') return c - 'A' + 10;
  else if(c >= 'a' && c <= 'f') return c - 'a' + 10;
  return -1;
}
// unsigned c; c -= '0'; if(c <= 9) return c; c -= 'A' - '0'; if(c <= 6) return c + 10; c -= 'a' - 'A'; if(c <= 6) return c + 10; return -1;

int inplaceHex2Array(unsigned char* buffer, unsigned char* sum)
{
  int count = 0;
  unsigned char* dst = buffer;
  while(1) {//*buffer) {
    int val;
    val = hex2int(*buffer++);
    if(val < 0) break; // for correct end of line detecting
    *dst = val<<4;
    val = hex2int(*buffer++);
    if(val < 0) return -1;
    *dst |= val;
    if(sum) sum += *dst;
    dst++;
    count++;
  }
  return count;
}

//#define	HEXPOS_LEN	1
//#define	HEXPOS_TYPE	4

int WriteAnyMemory(unsigned int addr, const unsigned char *data, unsigned int count)
{
  //   
  if(addr >= FLASH_START && addr + count - 1 <= FLASH_END)
  {
    //   
    if(!(addr >= BOOTV_END || addr + count - 1 <= BOOTV_START) )
    {
      //      
      //if(addr == BOOTV_START)
      //{
      //  // ,       
      //  if(&buff[1] != 0x82)
      //  {
      //    return -1;
      //  }
      //}
      ////     -     
      //     
      int nn, ret;
	  nn = (BOOTV_END - addr + 1);	// ,         
	  if(nn > count) nn = count;	//    ,      
      ret = WriteMemory(RESET_VECTOR_ADDR + (addr - BOOTV_START), data, nn);
      if(ret > 0) return ret;
      //    
	  if(count > nn)
	  {
        ret = WriteMemory(addr + nn, data + nn, count - nn);
        //ret = WriteMemory(BOOTV_END + 1, &buff[5] + nn, count - nn);
        if(ret > 0) return ret;
		return 0;
      }
    }
    else
    {
      //   ,   
      return WriteMemory(addr, data, count);//buff[0]);
    }
  }
  else if(addr >= EEPROM_START && addr + count - 1 <= EEPROM_END)
  {
    //   
    return WriteMemory(addr, data, count);
  }
  /*else if(addr >= RAM_START && addr + count - 1 <= RAM_END) //     ,   
  {
    //   
      int i;
      for(i = 0; i < count; ++i)
        ((unsigned char*)addr)[i] = data[i];
  }*/
  else if(addr >= OPTION_START && addr + count - 1 <= OPTION_END)
  {
    //   
    //for(i=0; i<DataCount; i++)
    //{
    //   FLASH_ProgramOptionByte((u32)(&DataAddress[i]), DataBuffer[i]);
    //}
  }

  //  
  return -1;
}

int ProcessWrite(unsigned char* buff)
{
  int n;
  unsigned char sum = 0;
  //if(buff[0] != ':') return -1;
  //  HEX    
  n = inplaceHex2Array(buff + 1, &sum);	//   ':'  .
  //     (    5       )
  if(n < 5 || (n - 5) != buff[1])
  {
    //xprintf("HEX: broken line\n", hextype, buff[0]);
    return -1;
  }
  //   
  if(sum)
  {
    //xprintf("HEX: wrong CRC token 0x%02x instead of 0x%02x\n", buff[n], (sum - buff[n])&0x00FF);
    return -1;
  }
  //   
  if(buff[4] > 5)
  {
    //xprintf("HEX: unknown record type %02x\n", buff[4]);
    return -1;
  }
  //    
  if(buff[4] == 1)
  {
    //   :00000001FF
    //if(buff[1] || buff[2] || buff[3]) return -1;
    return 0;
  }
  //    
  //if(!buff[4]) {} //     -  
  if(buff[4])
  {
    //     (   01  00)
    return -1;
  }
  //    
  return WriteAnyMemory((((unsigned int)buff[2]) << 8) | buff[3], &buff[5], buff[1]);//n - 5);
}

void DumpMemory(unsigned int addr, int count)
{
  int i;
  for(i = 0; i < count; ++i)
  {
    if(!(i & 0x0f) )
    {
      if(i)
      {
        puts("\n");
      }
      putVar(addr + i);
    }
    putc(' ');
    putHex(*(char*)(addr + i));
  }
  puts("\n");
}

void DumpAllFlash()
{
  DumpMemory(FLASH_START, FLASH_END - FLASH_START + 1);
  puts("ResetVector: ");
  DumpMemory(RESET_VECTOR_ADDR, 4);
}

void DumpAllEEPROM()
{
  DumpMemory(EEPROM_START, EEPROM_END - EEPROM_START + 1);
}

void DumpAllRAM()
{
  DumpMemory(RAM_START, RAM_END - RAM_START + 1);
}

void DumpAllOptions()
{
  DumpMemory(OPTION_START, OPTION_END - OPTION_START + 1);
}

#define is_space(c)	((c) == ' ' || (c) == '\t' || (c) == '\v' || (c) == '\r' || (c) == '\n')

void deinit()
{
    lock_memory();
    PC_ODR=0x00;
    PC_DDR=0x00;
    PC_CR1=0x00;
    PC_CR2=0x00;
    UART1_CR3 = 0;
    UART1_CR1 = 0;
    UART1_CR2 = 0;
    UART1_BRR2 = 0;
    UART1_BRR1 = 0;
    UART1_SR;
    UART1_DR;
}

void run()
{
    PD_ODR=0x00;
    PD_DDR=0x00;
    PD_CR1=0x00;
    PD_CR2=0x00;
    CLK_CKDIVR = 0x18;
    asm("LDW X,  SP ");
    asm("LD  A,  $FF");
    asm("LD  XL, A  ");
    asm("LDW SP, X  ");
    asm("JPF " RESET_VECTOR_ASM);
}

int main( void )
{
  CLK_CKDIVR = 0;     //    (Fsys = Fosc = 16MHz)
  //   
  PD_DDR_bit.DDR2 = 0; // Select mode
  PD_CR1_bit.C12 = 1;  //   Pull-up
  PD_CR2_bit.C22 = 0;  //  
  if( (PD_IDR & MASK_PD_IDR_IDR2) && ( *((const unsigned char*)RESET_VECTOR_ADDR) == 0x82) ) {
    run();
  }
  UART_init();
  puts("STM8 TopFlash bootloader.\n");
  while(1)
  {
    char *str;
    char c;
    str = gets();
    while(is_space(*str)) ++str;
    c = *str;
    if(c >= 'A' && c <= 'Z')
    {
      c -= 'A' - 'a';
    }
    if(c == 'q')
    {
      if(*((const unsigned char*)RESET_VECTOR_ADDR) == 0x82)
        break;      // quit
      puts("Firmware missed.");
    }
    switch(c)
    {
    case 0:
      break;	//  
	case ':':
      unlock_PROG();
      unlock_DATA();
      if( ProcessWrite((unsigned char*)str) )
      {
        puts("Error duaring process.\n");
      }
      else
      {
        puts("Ok.\n");
      }
      lock_memory();
      break;
    case 'f':
      // dump flash
      DumpAllFlash();
      break;
	case 'd':
      // dump eeprom
      DumpAllEEPROM();
      break;
	case 'r':
      // dump RAM
      DumpAllRAM();
      break;
	case 'o':
      // dump options
      DumpAllOptions();
      break;
    default:
      puts("Wrong token. Only f|d|q|r|o|: allowed.\n");
      break;
    }
  }
  deinit();
  return 0;
}

// For test
/*
empty
:00000001FF

Fill EEPROM up
:1040000000112233445566778899AABBCCDDEEFFB8
:00000001FF

Clear EEPROM up
:1040000000000000000000000000000000000000AF
:00000001FF

Fill EEPROM 0x20
:01402000346B
:00000001FF

Clear EEPROM 0x20
:01402000009F
:00000001FF
*/