/* Settings.h

Copyright (c) 2018 plBots@yandex.ru

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef SETTINGS_H_
#define SETTINGS_H_

#include <string>
#include <map>
#include <istream>

namespace ESD {
namespace Settings {

// --- ISettings ---

/// Interface to configuration. It will be easier to migrate from one storage type to another (plain text, XML, DB) with it.
class ISettings {
public:
	virtual ~ISettings() {};

	virtual bool Get(const std::string&, std::string&) const = 0;
	virtual bool Set(const std::string&, const std::string&) = 0;
	virtual bool Unset(const std::string&) = 0;
};


// --- SettingsDummy ---

/// Ignoring implementation. Can be useful in proxy classes.
template <bool ReadResult = false, bool WriteResult = false, bool DeleteResult = WriteResult>
class SettingsDummy : public ISettings {
public:
	virtual bool Get(const std::string&, std::string&) const { return ReadResult; }
	virtual bool Set(const std::string&, const std::string&) { return WriteResult; }
	virtual bool Unset(const std::string&) { return DeleteResult; }
};


// --- SettingsMap ---

/// Simple ISettings implementation based on std::map
class SettingsMap : public ISettings {
	typedef std::map<std::string, std::string> ValuesList;
public:
	SettingsMap();
	virtual ~SettingsMap();

	// ISettings implementation
	virtual bool Get(const std::string&, std::string&) const;
	virtual bool Set(const std::string&, const std::string&);
	virtual bool Unset(const std::string&);

	// Additional methods to set configuration
	void Clear();

private:
	ValuesList values;
};


// --- PlainKVFile ---

// Configuration provider class.
class PlainKVFile {
	PlainKVFile();
public:

	static bool Load(ISettings*, std::istream&);
	static bool Load(ISettings*, const std::string&);

	//void Save(std::ostream&);
	//void Save(const std::string&);
};

// --- GroupProxy ---

// Config proxy class. Helps to hide part of config hierarchy from object.
class GroupProxy : public ISettings {
public:
	GroupProxy(ISettings* settings_, const std::string& group_ = "", const std::string& separator_ = ".");
	virtual ~GroupProxy();

	// ISettings implementation
	virtual bool Get(const std::string&, std::string&) const;
	virtual bool Set(const std::string&, const std::string&);
	virtual bool Unset(const std::string&);

	// Extra abilities
	std::string GroupName() const;
	void GroupName(const std::string&);

	std::string GroupSeparator() const;
	void GroupSeparator(const std::string&);

	inline ISettings* Provider() const { return settings; }
	inline void Provider(ISettings* settings_) { settings = settings_; }

protected:
	ISettings* settings;
	std::string groupName;
	std::string groupSeparator;
};


// --- WriteProxy ---

// Config proxy class. Helps to prevent write operations in parent config.
class WriteProxy : public ISettings {
public:
	WriteProxy(const ISettings* settings_ = 0, ISettings* cache_ = 0);
	virtual ~WriteProxy();

	// ISettings implementation
	virtual bool Get(const std::string&, std::string&) const;
	virtual bool Set(const std::string&, const std::string&);
	virtual bool Unset(const std::string&);

	inline const ISettings* Provider() const { return settings; }
	inline void Provider(const ISettings* settings_) { settings = settings_; }

	inline ISettings* Cache() const { return cache; }
	inline void Cache(ISettings* cache_) { cache = cache_; }

private:
	const ISettings* settings;
	ISettings* cache;
};


// --- CachedProxy ---

// Stores previously gotten values in cache.
class CachedProxy : public WriteProxy {
public:
	CachedProxy(const ISettings* settings_, ISettings* cache_);
	~CachedProxy();
	// ISettings implementation
	virtual bool Get(const std::string&, std::string&) const;
};


// --- Getter ---

// Helps to get values of different types (not only std::string) with different default values.
class Getter {
public:
	Getter(const ISettings* settings_);

	bool Get(const std::string&, bool) const;
	int Get(const std::string&, int) const;
	long int Get(const std::string&, long int) const;
	unsigned long int Get(const std::string&, unsigned long int) const;
	double Get(const std::string&, double) const;
	std::string Get(const std::string&, const std::string&) const;
	std::string Get(const std::string&, const char*) const;	// required because of bool Get(const std::string&, bool) const;

	template <class T>
	T Get(const std::string& key) const { return Get(key, T()); }

	inline const ISettings* Provider() const { return settings; }
	inline void Provider(const ISettings* settings_) { settings = settings_; }

protected:
	bool TryGet(const std::string&, std::string&) const;

private:
	const ISettings* settings;
};


// --- Setter ---

// Helps to set values of different types.
class Setter {
public:
	Setter(ISettings* settings_);

	bool Set(const std::string&, bool) const;
	bool Set(const std::string&, int) const;
	bool Set(const std::string&, long int) const;
	bool Set(const std::string&, unsigned long int) const;
	bool Set(const std::string&, double) const;
	bool Set(const std::string&, const std::string&) const;
	bool Set(const std::string&, const char*) const;	// required because of bool Set(const std::string&, bool) const;

	bool Unset(const std::string&) const;

	inline ISettings* Provider() const { return settings; }
	inline void Provider(ISettings* settings_) { settings = settings_; }

private:
	ISettings* settings;
};


// --- Accessor ---

// Helps to get and set values of different types.
class Accessor : public Getter, public Setter {
public:
	Accessor(ISettings* settings_);

	inline ISettings* Provider() const { return Setter::Provider(); }
	inline void Provider(ISettings* settings_) { Getter::Provider(settings_); Setter::Provider(settings_); }
};


} /* namespace Settings */
} /* namespace ESD */
#endif /* SETTINGS_H_ */
