/* Settings.cpp

Copyright (c) 2018 plBots@yandex.ru

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#include <cstdlib>	// strtod,...
#include <cstdio>	// snprintf
#include <fstream>
//#include <algorithm>

#include "Settings.h"

namespace ESD {
namespace Settings {

// --- Support functions ---

//static void LeftTrim(std::string &str) {
//	str.erase(str.begin(), std::find_if(str.begin(), str.end(), std::not1(std::ptr_fun<int, int>(std::isspace))));
//}
//
//static void RightTrim(std::string &str) {
//	str.erase(std::find_if(str.rbegin(), str.rend(), std::not1(std::ptr_fun<int, int>(std::isspace))).base(), str.end());
//}

static void LeftTrim(std::string &str) {
	std::string::iterator it = str.begin();
	while (it != str.end() && std::isspace(*it))
		it++;
	str.erase(str.begin(), it);
}

static void RightTrim(std::string &str) {
	std::string::reverse_iterator rit = str.rbegin();
	while (rit != str.rend() && std::isspace(*rit))
		rit++;
	str.erase(rit.base(), str.end());
}

static void Trim(std::string &str) {
    LeftTrim(str);
	RightTrim(str);
}


// --- SettingsMap ---

SettingsMap::SettingsMap() {
}

SettingsMap::~SettingsMap() {
}

bool SettingsMap::Get(const std::string& key, std::string& val) const {
	ValuesList::const_iterator it = values.find(key);
    if( it == values.end() ) {
    	return false;
    }
	val = it->second;
	return true;
}

bool SettingsMap::Set(const std::string& key, const std::string& val) {
#if !defined(SETTINGSMAP_SAFESET)
	// The easiest way
	values[key] = val;
#else
	// Hard way (no empty string for key if assignment fails)
	std::pair<ValuesList::iterator,bool> ins = values.insert(std::make_pair(key,val));
	if(!ins.second) {
		(*(ins.first)).second = val;
	}
#endif
	return true;
}

bool SettingsMap::Unset(const std::string& key) {
	return values.erase(key) != 0;
}

void SettingsMap::Clear() {
	values.clear();
}


// --- PlainKVFile ---


bool PlainKVFile::Load(ISettings* settings, std::istream& is) {
	//bool err = false;
	if(!settings || !is) {
		return false;
	}
	std::string line;
	while (std::getline(is, line)) {
		// Ignore comments
		if (line.empty() || line[0] == '#') {
			continue;
		}
		//line.erase(std::find(line.begin(), line.end(), '#'), line.end());	// version of comment can start from any position
		size_t eqPos = line.find('=');
		if (eqPos == std::string::npos) {
			continue;
		}
		std::string key(line, 0, eqPos);
		std::string value(line, eqPos + 1, -1);
		//std::string value(line, eqPos != std::string::npos ? eqPos + 1 : line.length(), -1);
		Trim(key);
		Trim(value);
		if(!key.empty()) {
			settings->Set(key, value);
		}
	}
	return true;
}


bool PlainKVFile::Load(ISettings* settings, const std::string& fileName) {
	std::ifstream file(fileName.c_str());
	return Load(settings, file);
}


// --- GroupProxy ---

GroupProxy::GroupProxy(ISettings* settings_, const std::string& group_, const std::string& separator_) :
		settings(settings_), groupName(group_), groupSeparator(separator_)
{
}

GroupProxy::~GroupProxy() {
}

bool GroupProxy::Get(const std::string& key, std::string& val) const {
	ISettings* provider = this->Provider();
	return provider && provider->Get(this->groupName + this->groupSeparator + key, val);
}

bool GroupProxy::Set(const std::string& key, const std::string& val) {
	ISettings* provider = this->Provider();
	return provider && provider->Set(this->groupName + this->groupSeparator + key, val);
}

bool GroupProxy::Unset(const std::string& key) {
	ISettings* provider = this->Provider();
	return provider && provider->Unset(this->groupName + this->groupSeparator + key);
}

std::string GroupProxy::GroupName() const {
	return this->groupName;
}

void GroupProxy::GroupName(const std::string& group_) {
	this->groupName = group_;
}

std::string GroupProxy::GroupSeparator() const {
	return this->groupSeparator;
}

void GroupProxy::GroupSeparator(const std::string& separator_) {
	this->groupSeparator = separator_;
}


// --- WriteProxy ---

WriteProxy::WriteProxy(const ISettings* settings_, ISettings* cache_) :
		settings(settings_), cache(cache_)
{
}

WriteProxy::~WriteProxy() {
}

bool WriteProxy::Get(const std::string& key, std::string& val) const {
	ISettings* store = this->Cache();
	bool ret = (store && store->Get(key, val));
	if (!ret) {
		const ISettings* provider = this->Provider();
		ret = (provider && provider->Get(key, val));
	}
	return ret;
}

bool WriteProxy::Set(const std::string& key, const std::string& val) {
	ISettings* store = this->Cache();
	return store && store->Set(key, val);
}

bool WriteProxy::Unset(const std::string& key) {
	ISettings* store = this->Cache();
	return store && store->Unset(key);
}


// --- CachedProxy ---

CachedProxy::CachedProxy(const ISettings* settings_, ISettings* cache_) :
		WriteProxy(settings_, cache_)
{
}

CachedProxy::~CachedProxy() {
}

bool CachedProxy::Get(const std::string& key, std::string& val) const {
	ISettings* store = this->Cache();
	bool ret = (store && store->Get(key, val));
	if (!ret) {
		const ISettings* provider = this->Provider();
		ret = (provider && provider->Get(key, val));
		if (ret) {
			//Set(key, val);
			if (store) {
				store->Set(key, val);
			}
		}
	}
	return ret;
}


// --- Getter ---

Getter::Getter(const ISettings* settings_) : settings(settings_) {
}

inline bool Getter::TryGet(const std::string& key, std::string& val) const {
	const ISettings* provider = this->Provider();
	return provider && provider->Get(key, val);
}


bool Getter::Get(const std::string& key, bool def) const {
	int val = def;
	return Get(key, val) ? !!val : def;
}

int Getter::Get(const std::string& key, int def) const {
	std::string val;
	return TryGet(key, val) ? std::atoi(val.c_str()) : def;
}

long int Getter::Get(const std::string& key, long int def) const {
	std::string val;
#if !defined(SETTINGSMAP_STRICTCONVERSION)
	return TryGet(key, val) ? strtol(val.c_str(), NULL, 0) : def;
#else
	if (TryGet(key, val)) {
		char* end;
		const char* ptr = val.c_str();
		long int lVal = std::strtol(ptr, &end, 0);
		if (ptr != end && (!end || !*end || std::isspace(*end))) {
			return lVal;
		}
	}
	return def;
#endif
}

unsigned long int Getter::Get(const std::string& key, unsigned long int def) const {
	std::string val;
#if !defined(SETTINGSMAP_STRICTCONVERSION)
	return TryGet(key, val) ? strtoul(val.c_str(), NULL, 0) : def;
#else
	if (TryGet(key, val)) {
		char* end;
		const char* ptr = val.c_str();
		unsigned long int ulVal = std::strtoul(ptr, &end, 0);
		if (ptr != end && (!end || !*end || std::isspace(*end))) {
			return ulVal;
		}
	}
	return def;
#endif
}

double Getter::Get(const std::string& key, double def) const {
	std::string val;
#if !defined(SETTINGSMAP_STRICTCONVERSION)
	return TryGet(key, val) ? strtoul(val.c_str(), NULL, 0) : def;
#else
	if (TryGet(key, val)) {
		char* end;
		const char* ptr = val.c_str();
		double dVal = std::strtod(ptr, &end);
		if(ptr != end && (!end || !*end || std::isspace(*end))) {
			return dVal;
		}
	}
	return def;
#endif
}

std::string Getter::Get(const std::string& key, const std::string& def) const {
	std::string val;
	return TryGet(key, val) ? val : def;
}

std::string Getter::Get(const std::string& key, const char* def) const {
	std::string val;
	return TryGet(key, val) ? val : (def ? std::string(def) : std::string());
}



// --- Setter ---

Setter::Setter(ISettings* settings_) : settings(settings_) {
}

bool Setter::Set(const std::string& key, int val) const {
	char buff[16];
	int n = std::snprintf(buff, sizeof(buff)/sizeof(buff[0]), "%d", val);
	if(n < 0 || (size_t)n >= sizeof(buff)/sizeof(buff[0])) {
		return false;
	}
	return Set(key, buff);
}

bool Setter::Set(const std::string& key, long int val) const {
	char buff[32];
	int n = std::snprintf(buff, sizeof(buff)/sizeof(buff[0]), "%ld", val);
	if(n < 0 || (size_t)n >= sizeof(buff)/sizeof(buff[0])) {
		return false;
	}
	return Set(key, buff);
}

bool Setter::Set(const std::string& key, unsigned long int val) const {
	char buff[32];
	int n = std::snprintf(buff, sizeof(buff)/sizeof(buff[0]), "%lu", val);
	if(n < 0 || (size_t)n >= sizeof(buff)/sizeof(buff[0])) {
		return false;
	}
	return Set(key, buff);
}

bool Setter::Set(const std::string& key, double val) const {
	char buff[32];
	int n = std::snprintf(buff, sizeof(buff)/sizeof(buff[0]), "%f", val);
	if(n < 0 || (size_t)n >= sizeof(buff)/sizeof(buff[0])) {
		return false;
	}
	return Set(key, buff);
}

bool Setter::Set(const std::string& key, const std::string& val) const {
	ISettings* provider = this->Provider();
	return (provider && provider->Set(key, val));
}

bool Setter::Set(const std::string& key, const char* val) const {
	return Set(key, val ? std::string(val) : std::string());
}

bool Setter::Unset(const std::string& key) const {
	ISettings* provider = this->Provider();
	return (provider &&	provider->Unset(key));
}


// --- Accessor ---
Accessor::Accessor(ISettings* settings_) : Getter(settings_), Setter(settings_) {
	// nothing to do
}


} /* namespace Settings */
} /* namespace ESD */
