/*
 * sdcard.c
 *
 *  Created on: Jan 23, 2011
 *      Author: James
 */

#include "type.h"
#include "ssp.h"
#include "sdcard.h"

void delay(uint32_t ms);

#define CMD_GO_IDLE_STATE         0
#define CMD_SEND_OP_COND          1
#define CMD_SEND_CSD              9
#define CMD_SEND_CID              10
#define CMD_STOP_TRANSMISSION     12
#define CMD_SEND_STATUS           13
#define CMD_SET_BLOCKLEN          16
#define CMD_READ_SINGLE_BLOCK     17
#define CMD_READ_MULTIPLE_BLOCK   18
#define CMD_WRITE_SINGLE_BLOCK    24
#define CMD_WRITE_MULTIPLE_BLOCK  25
#define CMD_PROGRAM_CSD           27
#define CMD_SET_WRITE_PROT        28
#define CMD_CLR_WRITE_PROT        29
#define CMD_SEND_WRITE_PROT       30
#define CMD_TAG_SECTOR_START      32
#define CMD_TAG_SECTOR_END        33
#define CMD_UNTAG_SECTOR          34
#define CMD_TAG_ERASE_GROUP_START 35
#define CMD_TAG_ERASE_GROUP_END   36
#define CMD_UNTAG_ERASE_GROUP     37
#define CMD_ERASE                 38
#define CMD_LOCK_UNLOCK           42
#define CMD_APP_CMD               55
#define CMD_READ_OCR              58
#define CMD_CRC_ON_OFF            59
#define ACMD_SEND_OP_COND         41

#define IDLE_STATE		1
#define SD_DATA_SIZE	512

typedef union
{
   uint8_t Index[6];
   struct
      {
      uint8_t Command;
      uint32_t Argument;
      uint8_t Cksum;
   } CA;
} CommandStructure;

typedef union
{
   uint8_t b[4];
   uint32_t ul;
} b_ul;

void delay(uint32_t ms) {
	volatile int n;
	ms *= 8000;
	for(n = 0; n < ms; n++);
}

uint8_t SD_Command( uint8_t ThisCommand, uint32_t ThisArgument )
{
   b_ul Temp;
   int8_t i;

   /* enable the device... */
   ssp_enable_cs();

   /* send buffer clocks to insure no operations are pending... */
   ssp_transfer( 0xFF );

   /* send command */
   ssp_transfer(0x40 | ThisCommand);

   /* send argument */
   Temp.ul = ThisArgument;
   //for( i=0; i<4; i++ )
   for( i=3; i>-1; i-- )  // little-endian
	   ssp_transfer( Temp.b[ i ] );

   /* send CRC */
   ssp_transfer((ThisCommand == CMD_GO_IDLE_STATE)? 0x95:0xFF);

   /* send buffer clocks to insure card has finished all operations... */
   ssp_transfer( 0xFF );
   return( 0 );
}

uint8_t SD_GetR1()
{
   uint8_t i, j;

   for( i=0; i<8; i++ )
   {                        /* response will be after 1-8 0xffs.. */
      j = ssp_transfer( 0xff );
      if(j != 0xff)         /* if it isn't 0xff, it is a response */
         return(j);
   }
   return(j);
}

uint16_t SD_GetR2()
{
   uint16_t R2;

   R2 = ((SD_GetR1())<< 8)&0xff00;
   R2 |= ssp_transfer( 0xff );
   return( R2 );
}

uint16_t CardType;

uint8_t SD_Init()
{
   uint16_t CardStatus; // R2 value from status inquiry...
   uint16_t Count;      // local counter

   // Global CardType - b0:MMC, b1:SDv1, b2:SDv2

   /* initial speed is slow... */
   ssp_init( SSP_SLOW );

   /* disable SPI chip select... */
   ssp_disable_cs();

   /* fill send data with all ones - 80 bits long to   */
   /* establish link with SD card this fulfills the    */
   /* 74 clock cycle requirement...  */
   for(Count=0;Count<10;Count++)
	   ssp_transfer( 0xFF );

   /* enable the card with the CS pin... */
   ssp_enable_cs();

   /* ************************************************ */
   /* SET SD CARD TO SPI MODE - IDLE STATE...          */
   /* ************************************************ */
   Count = 1000;     // one second...
   CardType = 0;

   /* wait for card to enter IDLE state... */
   do
   {
	   delay(1);
      SD_Command( CMD_GO_IDLE_STATE, 0 );
   } while((SD_GetR1() != IDLE_STATE) && (--Count));

   /* timeout if we never made it to IDLE state... */
   if( !Count )
      return( SD_TIME_OUT );

   /* ************************************************ */
   /* COMPLETE SD CARD INITIALIZATION                  */
   /* FIGURE OUT WHAT TYPE OF CARD IS INSTALLED...     */
   /* ************************************************ */
   Count = 2000;     // two seconds...

   /* Is card SDSC or MMC? */
   SD_Command( CMD_APP_CMD, 0 );
   SD_Command( ACMD_SEND_OP_COND, 0 );
   if( SD_GetR1() <= 1 )
   {
      CardType = 2;
   }
   else
   {
      CardType = 1;
   }

   /* wait for initialization to finish... */
   do
   {
	   delay(1);
      if( CardType == 2 )
      {
         SD_Command( CMD_APP_CMD, 0 );
         SD_Command( ACMD_SEND_OP_COND, 0 );
      }
      else
      {
         SD_Command( CMD_SEND_OP_COND, 0 );
      }
   } while(SD_GetR1() && --Count);

   if( !Count )
      return( SD_TIME_OUT );

   /* ************************************************ */
   /* QUERY CARD STATUS...                             */
   /* ************************************************ */
   SD_Command( CMD_SEND_STATUS, 0 );
   CardStatus = SD_GetR2();

   if( CardStatus )
      return( SD_ERROR );

   /* ************************************************ */
   /* SET BLOCK SIZE...                                */
   /* ************************************************ */
   SD_Command( CMD_SET_BLOCKLEN, 512 );
   if( SD_GetR1() )
   {
      CardType = 0;
      return( SD_ERROR );
   }

   /* ************************************************ */
   /* SWITCH TO HIGHEST SPI SPEED...                   */
   /* ************************************************ */
   ssp_init( SSP_FAST );

   /* disable the card with the CS pin... */
   ssp_disable_cs();

   /* return OK... */
   return( 0 );
}

uint8_t SD_ReadSector( uint32_t SectorNumber, uint8_t *Buffer )
{
   uint8_t c, i;
   uint16_t count;

   /* send block-read command... */
   SD_Command( CMD_READ_SINGLE_BLOCK, SectorNumber << 9 );
   c = SD_GetR1();
   i = SD_GetR1();
   count = 0xFFFF;

   /* wait for data token... */
   while( (i == 0xff) && --count)
      i = SD_GetR1();

   /* handle time out... */
   if(c || i != 0xFE)
      return( 1 );

   /* read the sector... */
   for( count=0; count<SD_DATA_SIZE; count++)
      *Buffer++ = ssp_transfer(0xFF);

   /* ignore the checksum... */
   ssp_transfer(0xFF);
   ssp_transfer(0xFF);

   /* release the CS line... */
   ssp_disable_cs();

   return( 0 );
}


uint8_t SD_WriteSector( uint32_t SectorNumber, uint8_t *Buffer )
{
	uint8_t i;
	uint16_t count;

   /* send block-write command... */
   SD_Command( CMD_WRITE_SINGLE_BLOCK, SectorNumber << 9 );
   i = SD_GetR1();

   /* send start block token... */
   ssp_transfer( 0xFE );

   /* write the sector... */
   for( count= 0; count< 512; count++ )
   {
	   ssp_transfer(*Buffer++);
   }
   /* ignore the checksum (dummy write)... */
   ssp_transfer(0xFF);
   ssp_transfer(0xFF);

   /* wait for response token... */
   while( ssp_transfer( 0xFF ) != 0xFF)

   /* these 8 clock cycles are critical for the card */
   /* to finish up whatever it's working on at the */
   /* time... (before CS is released!) */
	   ssp_transfer( 0xFF );

   /* release the CS line... */
   ssp_disable_cs();
   ssp_transfer( 0xFF );
   return( 0 );
}

uint8_t SD_WaitForReady()
{
   uint8_t i;
   uint16_t j;

   ssp_transfer( 0xFF );

   j = 500;
   do
   {
      i = ssp_transfer( 0xFF );
      delay( 1 );
   } while ((i != 0xFF) && --j);

   return( i );
}
