/*
===============================================================================
 Name        : main.c
 Author      : 
 Version     :
 Copyright   : Copyright (C)
 Description : main definition
===============================================================================
*/

#ifdef __USE_CMSIS
#include "LPC13xx.h"
#endif

#include "gpio.h"
#include "lcd.h"

// LPCXpresso processor card LED
#define LED_PORT 0		// Port for led
#define LED_BIT	 7		// Bit on port for led
#define LED_ON	 1		// Level to set port to turn on led
#define LED_OFF	 0		// Level to set port to turn off led

#define BUTTON_PORT	2		/* Порт для подключия кнопки */
#define BUTTON_BIT	10		/* Номер бита в порту для кнопки */
#define BUTTON_UP	1		/* Кнопка отжата */
#define BUTTON_DOWN	0		/* Кнопка нажата */

#define ADC_CLK		4500000	/* для частоты преобразования 4.5МГц */
#define ADC_VOLTAGE	3150	/* Напряжение опоры АЦП в миливольтах */

// --- Средства работы со временем - Системный таймер ---
static volatile uint32_t msTicks = 0;		// counts 1ms timeTicks

void SysTick_Handler(void)
{
	msTicks++;		// инкремент счётчика времени
}

void delay_ms(uint32_t ms)
{
	uint32_t startTicks;
	startTicks = msTicks;
	while((msTicks - startTicks) < ms);	// Ждем завершения периода
}


// --- I2C ---
// Флаги регистра CONSET (I2C Control Set Register)
#define I2CONSET_I2EN		0x00000040
#define I2CONSET_AA			0x00000004
#define I2CONSET_SI			0x00000008
#define I2CONSET_STO		0x00000010
#define I2CONSET_STA		0x00000020
// Флаги регистра CONCLR (I2C Control clear Register)
#define I2CONCLR_AAC		0x00000004
#define I2CONCLR_SIC		0x00000008
#define I2CONCLR_STAC		0x00000020
#define I2CONCLR_I2ENC		0x00000040
// Время высокого уровня на линии SCL (значение для регистра SCLH)
#define I2SCLH_SCLH			0x00000180
// Время низкого уровня на линии SCL (значение для регистра SCLL)
#define I2SCLL_SCLL			0x00000180

int LM75_init()
{
	LPC_SYSCON->PRESETCTRL		|= (1<<1);	// Сняли Reset с I2C
	LPC_SYSCON->SYSAHBCLKCTRL	|= (1<<5);	// Разрешили тактирование

	LPC_IOCON->PIO0_4	&= ~0x3F;		// выбор функции I2C SCL для вывода P0.4
	LPC_IOCON->PIO0_4	|= 0x01;
	LPC_IOCON->PIO0_5	&= ~0x3F;		// выбор функции I2C SDA для вывода P0.5
	LPC_IOCON->PIO0_5	|= 0x01;

	// Сброс флагов контроллера I2C
	LPC_I2C->CONCLR	= I2CONCLR_AAC		// флаг Assert Acknowledge
					| I2CONCLR_SIC		// флаг наличия прерывания
					| I2CONCLR_STAC		// флаг генерации Start
					| I2CONCLR_I2ENC	// флаг разрешения работы
					;

	LPC_I2C->SCLL	= I2SCLL_SCLL;		// Время высокого уровня на линии SCL
	LPC_I2C->SCLH	= I2SCLH_SCLH;		// Время низкого уровня на линии SCL

	//NVIC_EnableIRQ(I2C_IRQn);			// Разрешаем обработку прерываний от I2C

	LPC_I2C->CONSET = I2CONSET_I2EN;	// Разрешаем работу I2C
	return 1;
}

int I2Cprocess(uint32_t set, uint32_t clear, uint32_t code) {
	// Устанавливаем биты управления
	LPC_I2C->CONSET = set;
	// Сбрасываем биты управления
	LPC_I2C->CONCLR = clear;
	// Ожидаем завершения операции
	while((LPC_I2C->CONSET & I2CONSET_SI) != I2CONSET_SI);
	// Проверяем результат и возвращаем код
	return (LPC_I2C->STAT == code) ? 1 : 0;
}

int16_t LM75_read(int num, int *ok)
{
	int16_t temp = 0;
	if(ok) *ok = 0;
	while(1) {
		// Устанавливаем флаг генерации Start
		if(!I2Cprocess(I2CONSET_STA, 0, 0x08)) break;
		// Занятие шины прошло успешно, помещаем адрес термометра с заданным номером и флаг чтения
		LPC_I2C->DAT	= 0x90 | ((num<<1)&0x0E) | 0x01;
		if(!I2Cprocess(0, I2CONCLR_STAC | I2CONCLR_SIC, 0x40)) break;
		// Устройство отозвалось продолжаем обмен и разрешаем подтверждение следующего принятого байта
		if(!I2Cprocess(I2CONSET_AA, I2CONCLR_SIC, 0x50)) break;
		// Нами было передано подтверждение, сохраняем старший байт результата
		temp |= (LPC_I2C->DAT)<<8;
		// продолжаем обмен и запрещаем подтверждение следующего принятого байта
		if(!I2Cprocess(0, I2CONCLR_AAC | I2CONCLR_SIC, 0x58)) break;
		// Нами был передан отказ, сохраняем младший байт результата
		temp |= (LPC_I2C->DAT)&0xE0;
		if(ok) *ok = 1;
		break;
	}
	// Завершаем обмен (нормально или при ошибке)
	LPC_I2C->CONSET = I2CONSET_STO;
	LPC_I2C->CONCLR = I2CONCLR_SIC;
	return temp;
}

// --- Основная программа ---

int main(void) {
	int num, ok;
	int16_t temp;
	char buffer[16] = "T0:!+199.999*C";
	int hi, lo;
	char sig;
	GPIOInit();
	GPIOSetDir(LED_PORT, LED_BIT, 1);
	GPIOSetValue(LED_PORT, LED_BIT, LED_OFF);
	GPIOSetDir(BUTTON_PORT, BUTTON_BIT, 0);
	SysTick_Config(SystemCoreClock / 1000);	// настройка таймера на период 1мс
	LM75_init();
	LCD_init();
	// Вывод приветствия
	LCD_gotoXY(1, 0);
	LCD_writeString("LM75  Thermo");
	LCD_gotoXY(2, 1);
	LCD_writeString("LPCXpresso");
	LCD_gotoXY(0, 2);
	LCD_writeString("Lesson I2C+SPI");
	LCD_gotoXY(2, 3);
	LCD_writeString("by Angel5a");
	LCD_gotoXY(0, 4);
	LCD_writeString("http://we.easyelectronics.ru");
	delay_ms(3000);
	while(1) {
		GPIOSetValue(LED_PORT, LED_BIT, LED_ON);
		for(num = 0; num < 6; num++) {
			temp = LM75_read(num, &ok);		// Измеряем значение очередного канала
			if(temp < 0) {
				hi = (-temp) >> 8;
				lo = (8 - (temp >> 5)) & 7;
				sig = '-';
			} else {
				hi = temp >> 8;
				lo = ((temp >> 5) & 7);
				sig = temp ? '+' : ' ';
			}
			lo *= 125;
			// Формирование результата
			buffer[1] = '0' + num;
			buffer[3] = ok ? ' ' : '!';
			// целая часть
			buffer[7] = '0' + hi % 10;
			buffer[6] = hi < 10 ? sig : '0' + hi / 10 % 10;
			buffer[5] = hi < 10 ? ' ' : hi < 100 ? sig : '0' + hi;
			buffer[4] = hi < 100 ? ' ' : sig;
			// дробная часть
			buffer[9] = '0' + lo / 100;
			buffer[10] = '0' + (lo / 10) % 10;
			buffer[11] = '0' + lo % 10;
			// Выводим результат
			LCD_gotoXY(0, num);
			LCD_writeString(buffer);
		}
		GPIOSetValue(LED_PORT, LED_BIT, LED_OFF);
		if(GPIOGetValue(BUTTON_PORT, BUTTON_BIT) != BUTTON_DOWN) {
			delay_ms(500);		// Если кнопка не нажата, добавляем задержку
		}
	}
	return 0 ;
}
