/*
===============================================================================
 Name        : main.c
 Author      : 
 Version     :
 Copyright   : Copyright (C)
 Description : main definition
===============================================================================
*/

#ifdef __USE_CMSIS
#include "LPC13xx.h"
#endif

#include <stdio.h>

#define PWM_RERIOD	2000	/* Период ШИМ в микросекундах */

// --- Средства работы со временем - Системный таймер ---
static volatile uint32_t msTicks = 0;		// counts 1ms timeTicks

void SysTick_Handler(void)
{
	msTicks++;		// инкремент счётчика времени
}

void delay_ms(uint32_t ms)
{
	uint32_t startTicks;
	startTicks = msTicks;
	while((msTicks - startTicks) < ms);	// Ждем завершения периода
}


// --- I2C ---

// Флаги регистра CONSET (I2C Control Set Register)
#define I2CONSET_I2EN		0x00000040
#define I2CONSET_AA			0x00000004
#define I2CONSET_SI			0x00000008
#define I2CONSET_STO		0x00000010
#define I2CONSET_STA		0x00000020

// Флаги регистра CONCLR (I2C Control clear Register)
#define I2CONCLR_AAC		0x00000004
#define I2CONCLR_SIC		0x00000008
#define I2CONCLR_STAC		0x00000020
#define I2CONCLR_I2ENC		0x00000040

// Время высокого уровня на линии SCL (значение для регистра SCLH)
#define I2SCLH_SCLH			0x00000180
// Время низкого уровня на линии SCL (значение для регистра SCLL)
#define I2SCLL_SCLL			0x00000180

int LM75_init()
{
	LPC_SYSCON->PRESETCTRL		|= (1<<1);	// Сняли Reset с I2C
	LPC_SYSCON->SYSAHBCLKCTRL	|= (1<<5);	// Разрешили тактирование

	LPC_IOCON->PIO0_4	&= ~0x3F;		// выбор функции I2C SCL для вывода P0.4
	LPC_IOCON->PIO0_4	|= 0x01;
	LPC_IOCON->PIO0_5	&= ~0x3F;		// выбор функции I2C SDA для вывода P0.5
	LPC_IOCON->PIO0_5	|= 0x01;

	// Сброс флагов контроллера I2C
	LPC_I2C->CONCLR	= I2CONCLR_AAC		// флаг Assert Acknowledge
					| I2CONCLR_SIC		// флаг наличия прерывания
					| I2CONCLR_STAC		// флаг генерации Start
					| I2CONCLR_I2ENC	// флаг разрешения работы
					;

	LPC_I2C->SCLL	= I2SCLL_SCLL;		// Время высокого уровня на линии SCL
	LPC_I2C->SCLH	= I2SCLH_SCLH;		// Время низкого уровня на линии SCL

	//NVIC_EnableIRQ(I2C_IRQn);			// Разрешаем обработку прерываний от I2C

	LPC_I2C->CONSET = I2CONSET_I2EN;	// Разрешаем работу I2C
	return 1;
}

int I2Cprocess(uint32_t set, uint32_t clear, uint32_t code) {
	// Устанавливаем биты управления
	LPC_I2C->CONSET = set;
	// Сбрасываем биты управления
	LPC_I2C->CONCLR = clear;
	// Ожидаем завершения операции
	while((LPC_I2C->CONSET & I2CONSET_SI) != I2CONSET_SI);
	// Проверяем результат и возвращаем код
	return (LPC_I2C->STAT == code) ? 1 : 0;
}

int16_t LM75_read(int num)
{
	int16_t temp = 0;
	while(1) {
		// Устанавливаем флаг генерации Start
		if(!I2Cprocess(I2CONSET_STA, 0, 0x08)) break;
		// Занятие шины прошло успешно, помещаем адрес термометра с заданным номером и флаг чтения
		LPC_I2C->DAT	= 0x90 | ((num<<1)&0x0E) | 0x01;
		if(!I2Cprocess(0, I2CONCLR_STAC | I2CONCLR_SIC, 0x40)) break;
		// Устройство отозвалось продолжаем обмен и разрешаем подтверждение следующего принятого байта
		if(!I2Cprocess(I2CONSET_AA, I2CONCLR_SIC, 0x50)) break;
		// Нами было передано подтверждение, сохраняем старший байт результата
		temp |= (LPC_I2C->DAT)<<8;
		// продолжаем обмен и запрещаем подтверждение следующего принятого байта
		if(!I2Cprocess(0, I2CONCLR_AAC | I2CONCLR_SIC, 0x58)) break;
		// Нами был передан отказ, сохраняем младший байт результата
		temp |= (LPC_I2C->DAT)&0xE0;
		break;
	}
	// Завершаем обмен (нормально или при ошибке)
	LPC_I2C->CONSET = I2CONSET_STO;
	LPC_I2C->CONCLR = I2CONCLR_SIC;
	return temp;
}

// --- Основная программа ---

int main(void) {
	int16_t temp;
	int hi, lo;
	SysTick_Config(SystemCoreClock / 1000);	// настройка таймера на период 1мс
	LM75_init();
	while(1) {
		temp = LM75_read(0);
		hi = temp >> 8;
		if(temp < 0) {
			lo = (8 - (temp >> 5)) & 7;
		} else {
			lo = ((temp >> 5) & 7);
		}
		lo *= 125;
		printf("Val = 0x%04x Temp = %d.%03d *C\n", temp, hi, lo);
		delay_ms(1000);
	}
	return 0 ;
}
