// Name		: usbdisplay.c
// Purpose	: USB Display based on USB virtual COM port Demo
// Version	: V0.1

#include "LPC13xx.h"    
#include "type.h"

#include "usb.h"
#include "usbcfg.h"
#include "usbhw.h"
#include "usbcore.h"
#include "cdc.h"
#include "cdcuser.h"
#include "lcd.h"
#include "lcdgui.h"
#include "mylib.h"

#define	EN_TIMER32_1	(1<<10)
#define	EN_IOCON		(1<<16)
#define	EN_USBREG		(1<<14)
#define	IOCON_COMMON_MODE_PULLDOWN	(1<<3)
#define IOCON_COMMON_MODE_PULLUP	(1<<4)

void ledSet(unsigned val) {
	if(val)	LPC_GPIO0->DATA |= 1<<7;
	else	LPC_GPIO0->DATA &= ~(1<<7);
}

// --- Средства работы со временем - Системный таймер ---

static volatile uint32_t msTicks = 0;		// counts 1ms timeTicks

void SysTick_Handler(void)
{
	msTicks++;		// инкремент счётчика времени
}

void delayms(uint32_t ms)
{
	uint32_t startTicks;
	startTicks = msTicks;
	if(startTicks > 0xFFFFFFFF - ms) {	// для случая "вблизи" верхней границы времени
		while(msTicks >= startTicks);
	}
	while((msTicks - startTicks) < ms);	// Ждем завершения периода
}

uint32_t getMS(void)
{
	return msTicks;
}

/// Устанавливает признак ошибки и блокирует продолжение выполнения кода
void fatalError(int code)
{
	(void) code; // отключаем предупреждение компилятора
	ledSet(1);
	while (1);
}

// --- Средства работы с дисплеем ---

void SPI_init() {	// Раздел 13.2 UM10375
    // Reset SSP (пункт 4)
	LPC_SYSCON->PRESETCTRL &= ~(1<<0);	// "маска" сброса SSP
	LPC_SYSCON->PRESETCTRL |= (1<<0);	// 1 в бит RST_SSP_0

    // Enable AHB clock to the SSP domain. (пункт 2)
    LPC_SYSCON->SYSAHBCLKCTRL |= (1<<11);

    // Divide by 1 (SSPCLKDIV also enables to SSP CLK) (пункт 3)
    LPC_SSP->CPSR = 0;	// отключим тактирование (а то мало ли)

    // Настройка портов ввода-вывода на SSP (пункт 1)
    // Set P0.8 to SSP MISO - мы пока не используем
    //IOCON_PIO0_8 &= ~IOCON_PIO0_8_FUNC_MASK;
    //IOCON_PIO0_8 |= IOCON_PIO0_8_FUNC_MISO0;
    // Set P0.9 to SSP MOSI
    LPC_IOCON->PIO0_9	&= ~(7<<0);
    LPC_IOCON->PIO0_9	|= (1<<0);	// использовать как вывод MOSI
    LPC_IOCON->PIO0_9	|= IOCON_COMMON_MODE_PULLUP;
    // Set 2.11 to SSP SCK (0.6 and 0.10 can also be used)
    LPC_IOCON->SCKLOC	= 1;	// SCK на вывод 2.11
    //LPC_IOCON->SCKLOC	= 0;	// SCK на вывод 0.10
    LPC_IOCON->PIO2_11	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
    LPC_IOCON->PIO2_11	|= (1<<0);	// использовать как вывод SCK
    LPC_IOCON->PIO2_11	|= IOCON_COMMON_MODE_PULLUP;
    //LPC_IOCON->JTAG_TCK_PIO0_10	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
    //LPC_IOCON->JTAG_TCK_PIO0_10	|= (2<<0);	// использовать как вывод SCK
    //LPC_IOCON->JTAG_TCK_PIO0_10	|= IOCON_COMMON_MODE_PULLUP;
    // Set P0.2/SSEL to GPIO output and high
    LPC_IOCON->PIO0_2	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
    LPC_IOCON->PIO0_2	|= (1<<0);	// использовать как вывод SSEL	(можно обычным GPIO как 0)
    LPC_IOCON->PIO0_2	|= IOCON_COMMON_MODE_PULLUP;
    LPC_GPIO0->DIR	|= 1<<2;
    LPC_GPIO0->DATA	|= 1<<2;
    //LPC_IOCON->PIO2_9	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
    //LPC_IOCON->PIO2_9	|= (0<<0);	// обычным GPIO как 0
    //LPC_IOCON->PIO2_9	|= IOCON_COMMON_MODE_PULLUP;
    //LPC_GPIO2->DIR	|= 1<<9;
    //LPC_GPIO2->DATA	|= 1<<9;

    // If SSP0CLKDIV = DIV1 -- (PCLK / (CPSDVSR X [SCR+1])) = (72,000,000 / (2 x [8 + 1])) = 4.0 MHz
    LPC_SSP->CR0	= ( (8<<0)	// Размер данных 1000 - 9 бит
					  | (0<<4)	// Формат фрейма 00 - SPI
					  | (0<<6)	// Полярность 0 - низкий уровень между фреймами
					  | (0<<7)	// Фаза 0 - по нарастанию
					  //| (8<<8)	// Делитель частоты шины на бит
					  | (3<<8)	// Делитель частоты шины на бит
					  ) ;

    // Clock prescale register must be even and at least 2 in master mode
    LPC_SSP->CPSR = 2;	// пердделитель 2-254 (кратно 2)

    // Clear the Rx FIFO (игнорируем)
    //uint8_t i, Dummy=Dummy;
    //for ( i = 0; i < SSP_FIFOSIZE; i++ ) { Dummy = SSP_SSP0DR; }
    // Enable the SSP Interrupt (не делаем, нам это не надо)
    //NVIC_EnableIRQ(SSP_IRQn);
    // Set SSPINMS registers to enable interrupts	(сказано же что не надо)
    //  enable all error related interrupts
    //SSP_SSP0IMSC = ( SSP_SSP0IMSC_RORIM_ENBL      // Enable overrun interrupt
    //               | SSP_SSP0IMSC_RTIM_ENBL);     // Enable timeout interrupt

    // Enable device and set it to master mode, no loopback (разрешаем работу)
    LPC_SSP->CR1	= ( (0<<0)	// 0 - Loop Back Mode Normal
					  | (1<<1)	// Разрешение работы 1 - разрешено
					  | (0<<2)	// Режим ведущий-ведомый 0 - мастер
					  );
}

void SPI_send(uint16_t value) {
	while ((LPC_SSP->SR & ((1<<1) | (1<<4))) != (1<<1));	// если буффер передачи не переполнен и устройство не занято
	//LPC_GPIO2->DATA	&= ~(1<<9);	// ncs = 0
	LPC_SSP->DR = value;
	//uint16_t Dummy = Dummy;
	//while ( (SSP_SSP0SR & (SSP_SSP0SR_BSY_BUSY|SSP_SSP0SR_RNE_NOTEMPTY)) != SSP_SSP0SR_RNE_NOTEMPTY );
	//Dummy = SSP_SSP0DR;
	//while ((LPC_SSP->SR & ((1<<1) | (1<<4))) != (1<<1));	// если буффер передачи не переполнен и устройство не занято
	//LPC_GPIO2->DATA	|= 1<<9;	// ncs = 1
}

void LCD_reset(void) {
	// Настройка для вывода Reset дисплея
	LPC_IOCON->PIO0_8	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
	LPC_IOCON->PIO0_8	|= IOCON_COMMON_MODE_PULLUP;
	LPC_GPIO0->DIR	|= 1<<8;
	LPC_GPIO0->DATA	|= 1<<8;
    //LPC_IOCON->PIO0_7	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
    //LPC_IOCON->PIO0_7	|= IOCON_COMMON_MODE_PULLUP;
    //LPC_GPIO0->DIR	|= 1<<7;
    //LPC_GPIO0->DATA	|= 1<<7;
    // Настройка для вывода Select
    LPC_IOCON->PIO0_2	&= ~(7<<0);	// Временно отключаем спецфункцию вывода выбора SPI
	delayms(100);
	// Сброс дисплея
	LPC_GPIO0->DATA	&= ~(1<<2);	// ncs = 0
	//LPC_GPIO2->DATA	&= ~(1<<9);	// ncs = 0
    LPC_GPIO0->DATA	&= ~(1<<8);	// nrst = 0
	//LPC_GPIO0->DATA	&= ~(1<<7);	// nrst = 0
    delayms(100);
    LPC_GPIO0->DATA	|= 1<<8;	// nrst = 1
	//LPC_GPIO0->DATA	|= 1<<7;	// nrst = 1
	LPC_GPIO0->DATA	|= 1<<2;	// ncs = 1
	//LPC_GPIO2->DATA	|= 1<<9;	// ncs = 1
	delayms(100);
	// Возврат спецфункции
	LPC_IOCON->PIO0_2	|= (1<<0);	// использовать как вывод SSEL
}

void PWM_init()
{
	// Разрешаем тактирование для таймера CT32B1
	LPC_SYSCON->SYSAHBCLKCTRL |= (1<<10);

	// Настраиваем PIO1.3 как вывод Timer1_32 MAT2 (и угораздило же SWDIO вывод использовать)
	//LPC_IOCON->ARM_SWDIO_PIO1_3	&= ~(7<<0);
    //LPC_IOCON->ARM_SWDIO_PIO1_3	|= (3<<0);	// использовать как вывод CT32B1MAT2
    //LPC_IOCON->ARM_SWDIO_PIO1_3	|= IOCON_COMMON_MODE_PULLUP;
	LPC_IOCON->JTAG_nTRST_PIO1_2	&= ~(7<<0);
    LPC_IOCON->JTAG_nTRST_PIO1_2	|= (3<<0);	// использовать как вывод CT32B1MAT1

	// Величина ШИМ
	//LPC_TMR32B1->MR2 = 0x7F;
	LPC_TMR32B1->MR1 = 0x7F;

	// Период ШИМ
	LPC_TMR32B1->MR0 = 0xFF;

	// Настройка режима совпадения
	LPC_TMR32B1->MCR = (2<<0);	// сброс при совпадении с MR0

    // Настраиваем на ШИМ (TOGLE + ENABLE)
	//LPC_TMR32B1->EMR =  (1<<2) | (3<<8);
	LPC_TMR32B1->EMR =  (1<<1) | (3<<6);

	// Запретить Timer1
	LPC_TMR32B1->TCR &= ~(1<<0);

	// Enable PWM0 and PWM2
	//LPC_TMR32B1->PWMC = (1<<0) | (1<<2);
	LPC_TMR32B1->PWMC = (1<<0) | (1<<1);

	// Make sure that the timer interrupt is enabled
	//NVIC_EnableIRQ(TIMER_32_1_IRQn);

	// Разрешить Timer1
	LPC_TMR32B1->TCR = (1<<0);
}

char* fmtSizeVar(uint32_t upval, uint32_t val)
{
	static char fmtSizeVarBuff[6];
	const char powChar[] = {' ', 'K', 'M', 'G', 'T', 'P'};
	int pow = 0;
	while(upval || val > 8*1024) {
		val = (upval << 22) | (val >> 10);
		upval >>= 10;
		pow++;
	}
	itoa(val, fmtSizeVarBuff, 10);
	int idx = 0;
	while(fmtSizeVarBuff[idx]) idx++;
	fmtSizeVarBuff[idx++] = powChar[pow];
	while(idx < 5) fmtSizeVarBuff[idx++] = ' ';
	fmtSizeVarBuff[idx] = 0;
	return &fmtSizeVarBuff[0];
}

#define STATISTICS_TOP	152
#define STATISTICS_GROUND	0xFFFFFFFF
#define STATISTICS_PASSIVE	0x00404040
#define ICONMAP(id)	&icons8x8[(id)<<3]
extern const uint8_t icons8x8[];

void UpdateStat()
{
	extern volatile uint32_t reciveTime;
	extern volatile uint32_t sendTime;
	extern volatile uint32_t lostTime;
	static uint32_t lastT = 0;
	uint32_t time = getMS();
	if( lastT <= time && lastT + 100 > time ) return;
	lastT = time;
	SetFont(FONT_MONOTYPE);
	SetGroundColor(STATISTICS_GROUND);
	int16_t x = 0;
	Fill(0, STATISTICS_TOP - 1, 128, 1, STATISTICS_GROUND);
	// connect status icon - disconnect, connect, configured, error - w = 8
	SetFaceColor(USB_Configuration ? 0x00800080 : STATISTICS_PASSIVE);
	DrawIcon8x8(x, STATISTICS_TOP, ICONMAP(USB_Configuration ? ICON_CONNECT : ICON_NOCONNECT));
	Fill(x + 8, STATISTICS_TOP, 2, 8, STATISTICS_GROUND);
	x += 10;
	// recive status icon - active, passive [, loose, error] - w = 8
	SetFaceColor(time < reciveTime + 500 ? 0x00004000 : STATISTICS_PASSIVE);
	DrawIcon8x8(x, STATISTICS_TOP, ICONMAP(ICON_RECIVE));
	Fill(x + 8, STATISTICS_TOP, 2, 8, STATISTICS_GROUND);
	x += 10;
	// send status icon - active, passive - w = 8
	SetFaceColor(time < sendTime + 500 ? 0x00404000 : STATISTICS_PASSIVE);
	DrawIcon8x8(x, STATISTICS_TOP, ICONMAP(ICON_SEND));
	Fill(x + 8, STATISTICS_TOP, 2, 8, STATISTICS_GROUND);
	x += 10;
	// recive errors - active, have, no - w = 8
	SetFaceColor(!lostTime ? STATISTICS_PASSIVE : time < lostTime + 2000 ? 0x00000080 : 0x00202080);
	DrawIcon8x8(x, STATISTICS_TOP, ICONMAP(ICON_STATUS));
	Fill(x + 8, STATISTICS_TOP, 2, 8, STATISTICS_GROUND);
	x += 10;
	// Disk status icon - active, passive, error [, nodisk] - w = 8 - reserved
	SetFaceColor(STATISTICS_PASSIVE);
	DrawIcon8x8(x, STATISTICS_TOP, ICONMAP(ICON_DISK));
	Fill(x + 8, STATISTICS_TOP, 2, 8, STATISTICS_GROUND);
	x += 10;
	// heart bet
	SetFaceColor(((getMS()/250) % 4) ? 0xFFFFFFFF : 0x000000FF);
	DrawIcon8x8(x, STATISTICS_TOP, ICONMAP(ICON_HEART3));
	int16_t remx = x + 8;
	// Clock icon - sync, nosynk, nodate - w = 8
	x = 128 - (8 + 2 /*+ 6*6*/ + 5*6 - 1);
	Fill(remx, STATISTICS_TOP, x - remx, 8, STATISTICS_GROUND);
	SetFaceColor(0x00000000);
	DrawIcon8x8(x, STATISTICS_TOP, ICONMAP(ICON_CLOCK));
	Fill(x + 8, STATISTICS_TOP, 2, 8, STATISTICS_GROUND);
	x += 10;
	// clock [+ date dd.mm + space] + time hh:mm - w = [5*6 +] 5*5 - 1
	char buff[16];
	SetFaceColor(0x00000000);
	DrawString(x, STATISTICS_TOP, ttoa(time, buff));
}


typedef struct _tagDisplayMessage {
	uint8_t start;
	uint8_t code;
	uint16_t id;
	uint8_t len;
	uint8_t reserved;
	uint8_t crc;
	uint8_t stop;
	uint8_t packet[56];
}DisplayMessage, *PDisplayMessage;
#define UDMSG_HEADERSIZE	8
#define UDMSG_START	0xAA
#define UDMSG_STOP	0x55
#define UDMSG_MAXLEN	sizeof(DisplayMessage)
#define UDMSG_CLEAR	0x36
#define UDMSG_LABEL	0x37
#define UDMSG_ICON	0x38
#define UDMSG_PROGRESS	0x39
#define UDMSG_SPEED	0x40
#define UDMSG_BAR	0x41
//#define UDMSG_

int UARTRead(void* buff, int size)
{
	int numAvailByte;
	CDC_OutBufAvailChar(&numAvailByte);
	if(!numAvailByte) return 0;
	if(numAvailByte > size) numAvailByte = size;
	return CDC_RdOutBuf(buff, &numAvailByte);
}

int pos = 0;
DisplayMessage msg;

void DispatchPacket()
{
	switch(msg.code) {
	case UDMSG_CLEAR:
		Clear(*(uint32_t*)&msg.packet[0]);
		break;
	case UDMSG_LABEL:
		Label((PLabelInfo)&msg.packet[0]);
		break;
	case UDMSG_ICON:
		Icon((PIconInfo)&msg.packet[0]);
		break;
	case UDMSG_PROGRESS:
		Progress((PProgressInfo)&msg.packet[0]);
		break;
	case UDMSG_SPEED:
		Speed((PSpeedView)&msg.packet[0]);
		break;
	case UDMSG_BAR:
		Bar((PBarView)&msg.packet[0]);
		break;
	default: break;
	}
	//Send answer
}

void ProcessCDC()
{
	while(1) {
		if(pos < UDMSG_HEADERSIZE)
			pos += UARTRead(((uint8_t*)&msg) + pos, UDMSG_HEADERSIZE - pos);
		if(pos < UDMSG_HEADERSIZE)
			return;
		if(msg.start == UDMSG_START && msg.stop == UDMSG_STOP && msg.len <= UDMSG_MAXLEN) {
			if(pos < msg.len)
				pos += UARTRead(((uint8_t*)&msg) + pos, msg.len - pos);
			if(pos < msg.len)
				return;
			uint8_t remCRC = msg.crc;
			msg.crc = 0;
			if(remCRC == crc8(((uint8_t*)&msg), msg.len, 0xFF))
			if(msg.reserved == 0x00)
			{
				DispatchPacket();
				pos = 0;
				return;
			}
			// контрольная сумма не совпала - неправильный пакет
			msg.crc = remCRC;
		}
		int i = 1;
		while(i < pos && ((uint8_t*)&msg)[i] != UDMSG_START)
			i++;
		pos -= i;
		if(pos != 0)
			memcpy(((uint8_t*)&msg), ((uint8_t*)&msg) + i, pos);
	}

}

/*----------------------------------------------------------------------------
  Main Program
 *---------------------------------------------------------------------------*/
int main (void) {
	/* Basic chip initialization is taken care of in SystemInit() called
	 * from the startup code. SystemInit() and chip settings are defined
	 * in the CMSIS system_<part family>.c file.
	 */

	/* Enable Timer32_1, IOCON, and USB blocks */
	LPC_SYSCON->SYSAHBCLKCTRL |= (EN_TIMER32_1 | EN_IOCON | EN_USBREG);

	USBIOClkConfig();

	// Настройка вывода порта светодиода
	LPC_SYSCON->SYSAHBCLKCTRL |= (1<<6);
  	LPC_IOCON->PIO0_7 = IOCON_COMMON_MODE_PULLDOWN;		// Pull Dn
  	LPC_GPIO0->DATA = 0;
  	LPC_GPIO0->DIR |= 1<<7;			// Вывод 0.7 на выход
  	LPC_IOCON->PIO3_0 = IOCON_COMMON_MODE_PULLUP;		// Pull Up
  	LPC_GPIO3->DIR &= ~(1<<0);		// Вывод 3.0 на вход
	if (SysTick_Config(SystemCoreClock / 1000)) {	// настройка таймера на период 1мс
		//while (1);
		fatalError(0);                      // Ошибка настройки
	}
	//NVIC_SetPriority(SysTick_IRQn, 1);		// задаем почти максимальный приоритет
	if ( !(SysTick->CTRL & SysTick_CTRL_CLKSOURCE_Msk) ) {
		LPC_SYSCON->SYSTICKCLKDIV = 0x08;	// из 16.6.1 замечание к внешнему таймеру (по примеру)
	}

	SPI_init();
	LCD_reset();
	LCD_init();
	Clear(0x00000000);
	PWM_init();

	SetFont(3);
	SetFaceColor(0xFFFFFFFF);
	SetGroundColor(0x00000000);
	DrawString(10, 0, "USB Display 1616");	// 69, 107 pixels

	CDC_Init();				// VCOM Initialization
	USB_Init();				// USB Initialization
	USB_Connect(TRUE);		// USB Connect

	while(1) {
		UpdateStat();
		if(!USB_Configuration) {
			__WFI();		// "sleep" if usb don't work
		} else {
			ProcessCDC();	// Work with usb subsystem
		}
	}
} // end main ()

