//------------------------------------------------------------------------------
// Файл		: lcd.c
// Создан	: 06.03.2011
// Автор	: Панзыга П.П.
// Описание	: Реализация средств для работы с ЖК-дисплеем.
// Copyright (c) 2011
//------------------------------------------------------------------------------

#include <stdint.h>
#include "lcd.h"
#include "font.h"

// Описание внешних функций
void delay(int ms);
void SPI_send(uint16_t value);

// Вспомогательные макросы
#define LCD_send(x)	SPI_send(x)
#define LCD_command(cmd)	LCD_send(cmd)
#define LCD_data(data) LCD_send(0x0100|(uint8_t)(data))

// Определение функций
void m_lcdResetClip(void);
void EndDraw(void);

// Описание переменных
RECT	m_lcdBound;		// Установленная для вывода область
POINT	m_lcdOutput;		// Очередная позиция для вывода
RECT	m_lcdClip;		// Размеры области отсечения
uint8_t	m_lcdClipOutput;	// Активны ли отсечения в текущей итерации вывода
const FONT_INFO *font = 0;	// Текущий шрифт для вывода
uint32_t colorFace;		// цвет текста/пера
uint32_t colorGround;		// цвет фона/заливки

// Nokia1616
#define displayOffsetX	2
#define displayOffsetY	1
#define displayWidth	128
#define displayHeight	160

const uint16_t init_lcd1616ph[] = {
		0xBA, 0x107, 0x115,	// Data Order
		0x25, 0x13F,		// Contrast
		0x11,				// Sleep Out
		0x13,				// Display Normal mode

		0x37,0x100,			// VSCROLL ADDR
		0x3A,0x105,			// COLMOD pixel format 4=12,5=16,6=18
		0x29,				// DISPON
		0x20,				// INVOFF
		0x13				// NORON
};

void LCD_init()
{
	const uint16_t *data = &init_lcd1616ph[0];
	uint16_t size = sizeof(init_lcd1616ph)/sizeof(init_lcd1616ph[0]);
	while(size--) {
		LCD_send(*data++);
	}
	LCD_command(0x2D);
	int i;
	for (i = 0; i < 32; i++)
		LCD_data(i<<1);
	for (i = 0; i < 64; i++)
		LCD_data(i);
	for (i = 0; i < 32; i++)
		LCD_data(i<<1);
	delay(100);
	//font = &verdana10ptFontInfo;
	m_lcdResetClip();
	EndDraw();
	SetFont(0);
}

uint32_t BeginDraw(int16_t left, int16_t top, uint16_t width, uint16_t height)
{
	int16_t right = left + width - 1;
	int16_t bottom = top + height - 1;
	if( left >= m_lcdClip.left && top >= m_lcdClip.top && right <= m_lcdClip.right && bottom <= m_lcdClip.bottom ) {	// RectInRect
		// область вывода полностью видима, используем "быстрый" вывод без проверок отсечения
		m_lcdClipOutput = 0;
	} else {
		// область вывода частично либо полностью невидима на дисплее, используем вывод с проверкой отсечений
		m_lcdClipOutput = 1;
		m_lcdOutput.x = left;
		m_lcdOutput.y = top;
		RECT_set(m_lcdBound, left, top, right, bottom);
		if(left < m_lcdClip.left) left = m_lcdClip.left;	// ClipRect
		if(top < m_lcdClip.top) top = m_lcdClip.top;
		if(right > m_lcdClip.right) right = m_lcdClip.right;
		if(bottom > m_lcdClip.bottom) bottom = m_lcdClip.bottom;
	}
	if( left > right || top > bottom ) {	// IsRectValid
		// область не видна на дисплее, завершаем вывод
		EndDraw();
		return 0;
	}
	uint32_t count;
	// Устанавливаем "виртуальные" границы вывода
	LCD_command(0x2A);	//LCD_command(CASETP);
	count	 = right - left + 1;
	left	+= displayOffsetX;
	right	+= displayOffsetX;
	LCD_data(left>>8);
	LCD_data(left);
	LCD_data(right>>8);
	LCD_data(right);
	// Диапозон строк
	LCD_command(0x2B);	//LCD_command(PASETP);
	count	*= (bottom - top + 1);
	top		+= displayOffsetY;
	bottom	+= displayOffsetY;
	LCD_data(top>>8);
	LCD_data(top);
	LCD_data(bottom>>8);
	LCD_data(bottom);
	LCD_command(0x2C);	//LCD_command(RAMWR);
	// Возвращаем количество видимых пикселей
	return count;
}

void EndDraw()
{
	m_lcdClipOutput = 1;
	m_lcdBound.right = m_lcdBound.left - 1;	// SetRect
}

void NextPoint(uint32_t color)
{
	// Надо ли проверять отсечения при выводе
	if( m_lcdClipOutput ) {
		int16_t x = m_lcdOutput.x;
		int16_t y = m_lcdOutput.y;
		// учит.отсеч.
		//if(!m_lcdClip.width /* || !m_lcdClip.height */) return;
		if(m_lcdBound.right < m_lcdBound.left) { // IsRectValid вывод недоступен
			return;
		}
		// Смещение на следующую позицию вывода
		if( m_lcdOutput.x >= m_lcdBound.right ) {
			// Если в конце строки - переходим на следующую
			m_lcdOutput.x = m_lcdBound.left;
			if( m_lcdOutput.y >= m_lcdBound.bottom ) {
				// Если последняя точка, переходим в начало (или можно завершить вывод вызовом EndDraw)
				m_lcdOutput.y = m_lcdBound.top;
			} else {
				m_lcdOutput.y++;
			}
		} else {
			m_lcdOutput.x++;
		}
		if(!(x >= m_lcdClip.left && x <= m_lcdClip.right && y >= m_lcdClip.top && y <= m_lcdClip.bottom) ) {	// PtInRect
			return;
		}
	}
	// точка будет видимой
	uint8_t r = color;
	uint8_t g = color>>8;
	uint8_t b = color>>16;
	LCD_data((r&0xF8)|(g>>5));
	LCD_data(((g<<3)&0xE0)|(b>>3));
}

void m_lcdSetClip(const RECT *r)
{
	// Проверяем на видимость отсечения
	RECT_set(m_lcdClip, r->left, r->top, r->right, r->bottom);
	// Устанавливаем "абсолютное" ограничение для вывода c усечением до границ дисплея
	if(m_lcdClip.left	< 0) m_lcdClip.left = 0;	// ClipRect
	if(m_lcdClip.top	< 0) m_lcdClip.top = 0;
	if(m_lcdClip.right	>= displayWidth) m_lcdClip.right = displayWidth - 1;
	if(m_lcdClip.bottom	>= displayHeight) m_lcdClip.bottom = displayHeight - 1;
	// Is clip valid
	if(m_lcdClip.bottom < m_lcdClip.top || m_lcdClip.right < m_lcdClip.left) {	// IsRectValid
		// invalid clip region
		RECT_set(m_lcdClip, 0, 0, -1, -1);
	}
}

void m_lcdResetClip()
{
	RECT_set(m_lcdClip, 0, 0, displayWidth - 1, displayHeight - 1);
}

uint16_t GetDisplayWidth()
{
	return displayWidth;
}

uint16_t GetDisplayHeight()
{
	return displayHeight;
}

void Fill(int16_t left, int16_t top, uint16_t width, uint16_t height, uint32_t color)
{
	uint32_t count;
	count = BeginDraw(left, top, width, height);
	m_lcdClipOutput = 0;	// Принудительно меняем функцию вывода, выводим только необходимое количество пикселей
	while(count--) {
		NextPoint(color);
	}
	EndDraw();
}

void Clear(uint32_t color)
{
	Fill(0, 0, displayWidth, displayHeight, color);
}

void Pixel(int16_t x, int16_t y, uint32_t color)
{
	Fill(x, y, 1, 1, color);
}

void DrawImage(const uint8_t *data, uint16_t len)
{
	uint8_t n;
	uint8_t r, g, b;
	while(len > 3) {
		n = *data;
		len--; data++;
		if(n & 0x80) {
			// Повторение одного пикселей одного цвета
			n = (n & 0x7f) + 1;
			r = data[0];
			g = data[1];
			b = data[2];
			len -= 3; data += 3;
			while(n--) {
				NextPoint(RGB(r, g, b));
			}
		} else {
			// Пиксели разного цвета
			n = (n) + 1;
			while(n-- && len >= 3) {
				r = data[0];
				g = data[1];
				b = data[2];
				len -= 3; data += 3;
				NextPoint(RGB(r, g, b));
			}

		}
	}
}

void DrawIcon8x8(int16_t x, int16_t y, const uint8_t *data)
{
	if(!data) return;
	// Установка области вывода
	if(BeginDraw(x, y, 8, 8)) {
		uint8_t i, j, mask;
		for(i = 0; i < 8; ++i) {
			mask = *data++;
			for(j = 0; j < 8; ++j) {
				NextPoint( (mask & 0x80) ? colorFace : colorGround );
				mask<<=1;
			}
		}
		EndDraw();
	}
}

void SetFont(uint8_t num)
{
	switch(num) {
	case 0:	font = &font5x8pxFontInfo;	break;
	case 1:	font = &ubuntu10ptFontInfo;	break;
	case 2:	font = &ubuntu10ptBoldFontInfo;	break;
	case 3:	font = &verdana10ptFontInfo;	break;
	default:	font = 0;
	}
}

void SetFaceColor(uint32_t color)
{
	colorFace = color;
}

void SetGroundColor(uint32_t color)
{
	colorGround = color;
}

#define CHAR_WIDTH(c)	(font->u8Flags & FONT_FIXEDWIDTH ? ((int)font->asFontCharInfo)&0x0FF : font->asFontCharInfo[(c) - font->u8FirstChar].width)
#define CHAR_SPACE		(font->u8Flags & FONT_FIXEDWIDTH ? 1 : 2)
#define CHAR_BITMAP(c)	(&font->au8FontTable[font->u8Flags & FONT_FIXEDWIDTH ? (c - font->u8FirstChar)*(((int)font->asFontCharInfo)&0x0FF) : font->asFontCharInfo[c - font->u8FirstChar].start])

uint8_t GetCharWidth(char c)
{
	if(!font) return 0;
	if(c < font->u8FirstChar || c > font->u8LastChar) return 0;
	return CHAR_WIDTH(c);
}

uint8_t GetFontHeight()
{
	if(!font) return 0;
	return font->u8Height;
}

uint16_t GetStringWidth(const char* str)
{
	if(!font || !str) return 0;
	uint8_t w = 0;
	uint16_t len = 0;
	while(*str) {	// Без проверки переполнения len
		if(w) len += CHAR_SPACE;
		w = GetCharWidth(*str);
		len += w;
		str++;
	}
	return len;
}

uint8_t DrawChar(int16_t x, int16_t y, uint8_t c)
{
	if(!font) return 0;
	if(c < font->u8FirstChar || c > font->u8LastChar) return 0;
	// Получаем информацию о символе
	uint8_t width = CHAR_WIDTH(c);
	uint8_t height = font->u8Height;
	// Установка области вывода
	if(BeginDraw(x, y, width, height)) {
		const uint8_t *ptr = CHAR_BITMAP(c);
		// вывод символа
		uint8_t i, j, k, mask;
		k = 0;
		for(i = 0; i < height; ++i) {
			for(j = 0; j < width; ++j) {
				if(!(k&0x07) ) mask = *ptr++;
				NextPoint( (mask & 0x80) ? colorFace : colorGround );
				mask<<=1;
				k++;
			}
			if(!(font->u8Flags & FONT_PACKEDDATA) ) k = 0;
		}
		EndDraw();
	}
	return width;
}

uint16_t DrawString(int16_t x, int16_t y, const char *str)
{
	if(!font || !str) return 0;
	int16_t remx = x;
	uint8_t w = 0;
	uint8_t height = font->u8Height;
	//if(y > displayHeight - height) return 0;
	while(*str) {
		if(w) {
			// Промежуток между символами
			Fill(x, y, CHAR_SPACE, height, colorGround);
			x += CHAR_SPACE;
		}
		if(x > displayWidth) break;
		w = DrawChar(x, y, *str);
		x += w;
		//if(!w && GetCharWidth(*str)) { break; } // ушли за границу экрана
		str++;
	}
	return x - remx;
}

uint16_t DrawText(const RECT *r, uint16_t flags, const char* str)
{
	uint16_t count;
	int16_t x, y;
	if(!r) return 0;
	m_lcdSetClip(r);
	uint16_t w, h;
	w = GetStringWidth(str);
	h = GetFontHeight(str);
	// учёт флагов вывода и т.д.
	if(flags & DT_CENTER) {	// Выравнивание по центру
		x = (r->right + r->left + 1 - w) / 2;
	} else if(flags & DT_RIGHT) {	// Выравнивание по правой стороне
		x = r->right + 1 - w;
	} else {	// Выравнивание по левой стороне (по умолчанию)
		x = r->left;
	}
	if(flags & DT_VCENTER) {	// Выравнивание по центру по вертикали
		y = (r->bottom + r->top + 1 - h) / 2;
	} else if(flags & DT_BOTTOM) {	// Выравнивание к верху
		y = r->bottom + 1 - h;
	} else {	// Выравнивание к низу (по умолчанию)
		y = r->top;
	}
	if(flags & DT_FILL) {
		// Медленно, но для начала пойдет
		if(r->top < y) Fill(r->left, r->top, r->right - r->left + 1, y - r->top, colorGround);	// верх
		if(r->left < x) Fill(r->left, y, x - r->left, h, colorGround); // лево
	}
	count = DrawString(x, y, str);
	if(flags & DT_FILL) {
		// Медленно, но для начала пойдет
		if(r->right >= x + w) Fill(x + w, y, r->right - (x + w) + 1, h, colorGround);	// право
		if(r->bottom >= y + h) Fill(r->left, y + h, r->right - r->left + 1, r->bottom - (y + h) + 1, colorGround); // низ
	}
	m_lcdResetClip();
	return count;
}

