﻿#include "LPC13xx.h"    
#include "type.h"

#define	IOCON_COMMON_MODE_PULLDOWN	(1<<3)
#define IOCON_COMMON_MODE_PULLUP	(1<<4)

void SPI_init() {	// Раздел 13.2 UM10375
    // Reset SSP (пункт 4)
    LPC_SYSCON->PRESETCTRL &= ~(1<<0);	// "маска" сброса SSP
    LPC_SYSCON->PRESETCTRL |= (1<<0);	// 1 в бит RST_SSP_0

    // Enable AHB clock to the SSP domain. (пункт 2)
    LPC_SYSCON->SYSAHBCLKCTRL |= (1<<11);

    // Divide by 1 (SSPCLKDIV also enables to SSP CLK) (пункт 3)
    LPC_SSP->CPSR = 0;	// отключим тактирование (а то мало ли)

    // Настройка портов ввода-вывода на SSP (пункт 1)
    // Set P0.9 to SSP MOSI
    LPC_IOCON->PIO0_9	&= ~(7<<0);
    LPC_IOCON->PIO0_9	|= (1<<0);	// использовать как вывод MOSI
    LPC_IOCON->PIO0_9	|= IOCON_COMMON_MODE_PULLUP;

    // Set 2.11 to SSP SCK (0.6 and 0.10 can also be used)
    LPC_IOCON->SCKLOC	= 1;	// SCK на вывод 2.11
    LPC_IOCON->PIO2_11	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
    LPC_IOCON->PIO2_11	|= (1<<0);	// использовать как вывод SCK
    LPC_IOCON->PIO2_11	|= IOCON_COMMON_MODE_PULLUP;

    // Set P0.2/SSEL to GPIO output and high
    LPC_IOCON->PIO0_2	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
    LPC_IOCON->PIO0_2	|= (1<<0);	// использовать как вывод SSEL	(можно обычным GPIO как 0)
    LPC_IOCON->PIO0_2	|= IOCON_COMMON_MODE_PULLUP;
    LPC_GPIO0->DIR	|= 1<<2;
    LPC_GPIO0->DATA	|= 1<<2;

    // If SSP0CLKDIV = DIV1 -- (PCLK / (CPSDVSR X [SCR+1])) = (72,000,000 / (2 x [3 + 1])) = 9.0 MHz
    LPC_SSP->CR0	= ( (8<<0)	// Размер данных 1000 - 9 бит
					  | (0<<4)	// Формат фрейма 00 - SPI
					  | (0<<6)	// Полярность 0 - низкий уровень между фреймами
					  | (0<<7)	// Фаза 0 - по нарастанию
					  | (3<<8)	// Делитель частоты шины на бит
					  ) ;

    // Clock prescale register must be even and at least 2 in master mode
    LPC_SSP->CPSR = 2;	// пердделитель 2-254 (кратно 2)

    // Enable device and set it to master mode, no loopback (разрешаем работу)
    LPC_SSP->CR1	= ( (0<<0)	// 0 - Loop Back Mode Normal
					  | (1<<1)	// Разрешение работы 1 - разрешено
					  | (0<<2)	// Режим ведущий-ведомый 0 - мастер
					  );
}

void SPI_send(uint16_t value) {
	while ((LPC_SSP->SR & ((1<<1) | (1<<4))) != (1<<1));	// если буффер передачи не переполнен и устройство не занято
	LPC_SSP->DR = value;
}

void LCD_reset(void) {
	// Настройка для вывода Reset дисплея
	LPC_IOCON->PIO0_8	&= ~(7<<0);	// сброс текущей функции порта ввода-вывода
	LPC_IOCON->PIO0_8	|= IOCON_COMMON_MODE_PULLUP;
	LPC_GPIO0->DIR	|= 1<<8;
	LPC_GPIO0->DATA	|= 1<<8;
	LPC_IOCON->PIO0_2	&= ~(7<<0);	// Временно отключаем спецфункцию вывода выбора SPI
	delayms(100);
	// Сброс дисплея
	LPC_GPIO0->DATA	&= ~(1<<2);	// ncs = 0
	LPC_GPIO0->DATA	&= ~(1<<8);	// nrst = 0
	delayms(100);
	LPC_GPIO0->DATA	|= 1<<8;	// nrst = 1
	LPC_GPIO0->DATA	|= 1<<2;	// ncs = 1
	delayms(100);
	// Возврат спецфункции
	LPC_IOCON->PIO0_2	|= (1<<0);	// использовать как вывод SSEL
}

void PWM_init()
{
	// Разрешаем тактирование для таймера CT32B1
	LPC_SYSCON->SYSAHBCLKCTRL |= (1<<10);

	// Настраиваем PIO1.3 как вывод Timer1_32 MAT2 (и угораздило же SWDIO вывод использовать)
	LPC_IOCON->JTAG_nTRST_PIO1_2	&= ~(7<<0);
	LPC_IOCON->JTAG_nTRST_PIO1_2	|= (3<<0);	// использовать как вывод CT32B1MAT1

	// Величина ШИМ
	LPC_TMR32B1->MR1 = 0x7F;

	// Период ШИМ
	LPC_TMR32B1->MR0 = 0xFF;

	// Настройка режима совпадения
	LPC_TMR32B1->MCR = (2<<0);	// сброс при совпадении с MR0

	// Настраиваем на ШИМ (TOGLE + ENABLE)
	LPC_TMR32B1->EMR =  (1<<1) | (3<<6);

	// Запретить Timer1
	LPC_TMR32B1->TCR &= ~(1<<0);

	// Enable PWM0 and PWM2
	LPC_TMR32B1->PWMC = (1<<0) | (1<<1);

	// Разрешить Timer1
	LPC_TMR32B1->TCR = (1<<0);
}


int main (void) {
	SPI_init();
	LCD_reset();
	LCD_init();
	Clear(0x00000000);
	PWM_init();

	SetFont(3);
	SetFaceColor(0xFFFFFFFF);
	SetGroundColor(0x00000000);
	DrawString(10, 0, "USB Display 1616");	// 69, 107 pixels

	while(1) __WFI();
}
