#include "iostm8s103f3.h"
#include "intrinsics.h"

//  1, Flash 8, EEPROM 640 
//  UART1, SPI1, I2C1, TIM1, TIM2, TIM4, BEEP, AWU, WatchDog, ADC1

void _delay_ms(int ms) {
  char i, j;
  while(ms--)
    for(i = 12; i; --i)
      for(j = 255; j; --j)
        asm("nop");
}

void HSE_init()
{
  //    ( )
  CLK_ECKR_bit.HSEEN  = 1;      //  HSE
  while(!CLK_ECKR_bit.HSERDY);  //    
  CLK_SWCR_bit.SWEN   = 1;      //     
  CLK_SWR             = 0xB4;   //  HSE   
  CLK_CKDIVR          = 0;      //   
  while(!CLK_SWCR_bit.SWIF);    //   
  CLK_SWCR_bit.SWIF   = 0;
}

void UART_init()
{
  //         
  // Table 54 RM0016 contain precalculated UART deviders
  // DIV = 0x008B -> BRR1=0x08, BRR2 = 0x0B // 115200 at 16MHz
  // DIV = 0x0693 -> BRR1=0x69, BRR2 = 0x03 //   9600 at 16MHz
  UART1_CR1_bit.M     = 0;  // 8 bit dataframe
  UART1_CR1_bit.PCEN  = 0;  //  no paryty in frame (all frame bits are data)
  UART1_CR3_bit.STOP  = 0;  // 1 stop bit
  UART1_BRR2  = 0x0B;       // Note: BRR2 set before BRR1
  UART1_BRR1  = 0x08;       
  //UART1_BRR2  = ((unsigned char)(coef >> 8) & 0xF0) | ((unsigned char)coef & 0x0F);
  //UART1_BRR2  = ((unsigned char)(coef >> 4) & 0xFF);
  UART1_CR2_bit.TEN   = 0;  // Disable transmitter
  UART1_CR2_bit.REN   = 1;  // Enable reciver
  UART1_CR2_bit.RIEN  = 1;  // Enable recive interrupt
}

// Warning! Change the LCD_init_hw function if you set different pins
//  HW SPI pins is: pc5 - sck, pc6 - mosi, pc7 - miso
#define LCD_SCK	PC_ODR_bit.ODR5
#define LCD_DAT	PC_ODR_bit.ODR6
#define LCD_SEL	PA_ODR_bit.ODR3
#define LCD_RES	PC_ODR_bit.ODR4

void LCD_init_hw()
{
  LCD_SEL = 1;
  PA_DDR_bit.DDR3 = 1; // SSEL
  PA_CR1_bit.C13 = 1;  //   Push-pull
  PA_CR2_bit.C23 = 1;  //   -  10 .
  
  LCD_RES = 1;
  PC_DDR_bit.DDR4 = 1; // RES
  PC_CR1_bit.C14 = 1;  //   Push-pull
  PC_CR2_bit.C24 = 1;  //   -  10 .

  LCD_SCK = 0;
  PC_DDR_bit.DDR5 = 1; // SCK
  PC_CR1_bit.C15 = 1;  //   Push-pull
  PC_CR2_bit.C25 = 1;  //   -  10 .
  
  //LCD_DAT = 1;
  PC_DDR_bit.DDR6 = 1; // MOSI
  PC_CR1_bit.C16 = 1;  //   Push-pull
  PC_CR2_bit.C26 = 1;  //   -  10 .
}

void LCD_reset(void)
{
  _delay_ms(100);
  //  
  LCD_SEL = 0;
  LCD_RES = 0;
  _delay_ms(100);
  LCD_RES = 1;
  LCD_SEL = 1;
  _delay_ms(100);
}

void LCD_send ( unsigned char data )
{
  int i;
  LCD_SEL = 0;
  LCD_SCK = 1;
  for(i = 0; i < 8; i++) {
    LCD_SCK = 0;
    LCD_DAT = (data & 0x80) ? 1 : 0;
    LCD_SCK = 1;
    data <<= 1;
  }
  LCD_SCK = 0;
  LCD_SEL = 1;
}

void LCD_cmd(unsigned char cmd)
{
  LCD_DAT = 0;
  LCD_send(cmd);
}

void LCD_data(unsigned char data)
{
  LCD_DAT = 1;
  LCD_send(data);
}

void LCD_init()
{
	LCD_init_hw();

	LCD_cmd( 0xE2 );  // Soft reset
	_delay_ms(10);
	  LCD_cmd( 0x3D );  // Charge pump
	  LCD_cmd( 0x01 );  // Charge pump = 4 (default 5 is too hight for 3.0 volt)
	  LCD_cmd( 0xE1 );  // Additional VOP for contrast increase
	  LCD_cmd( 0x16 );  // from -127 to +127
	LCD_cmd( 0xA4 );  // Power save OFF
	LCD_cmd( 0x2F );  // Booster ON Voltage regulator ON Voltage follover ON
	LCD_cmd( 0xAF );  // LCD on
}

#define INCREMENT_MODE  0x01
#define DISPLAY_SHIFT   0x02
#define DISPLAY_ON      0x04
#define CURSOR_MOVE     0x08
#define CURSOR_ON       0x10
#define CURSOR_BLINK    0x20
#define SHIFT_RIGHT     0x40

#define DDRAM_SIZE      128

unsigned char control;
unsigned char address;
unsigned char cursor;
unsigned char shift;
unsigned char state;
unsigned char CGRAM[8*5];
unsigned char DDRAM[DDRAM_SIZE]; 
unsigned char curBlink = 0;
const char Header[] = "-=LCDTerm emul=-";

#include "font5x8.h"

void HD44780_ScreenRedraw() //  
{
  int i, j, dataPos;
  unsigned char c, add;
  const unsigned char *ptr;
  LCD_cmd(0xB0);
  LCD_cmd(0x10);
  LCD_cmd(0x00);
  for(i = 0; i < 16; ++i)
  {
    c = Header[i];
    ptr = &font[5*(unsigned int)c];
    for(j = 0; j < 5; j++) {
      LCD_data(ptr[j]);
    }
    LCD_data(0x00);
  }
  for(i = 0; i < 16*6; ++i)
    LCD_data(0x10);
  for(i = 0; i < 6*16; ++i)
  {
 	//pos = (row % 2) * 64 + (row / 2) * 16 + col;	//  DCOLS == 16 && DROWS == 4
	//pos = (row % 2) * 64 + (row / 2) * 20 + col;	//  
    dataPos = ((i/16) % 2) * 64 + ((i/16) / 2) * 20 + (i%16);
    c = DDRAM[dataPos];
    if(c < 8) {
      ptr = &CGRAM[c * 5];
    } else {
      ptr = &font[5*(unsigned int)c];
    }
    if((control & CURSOR_ON) && dataPos == cursor) {
      add = !(control & CURSOR_BLINK) ? 0x80 : (curBlink & 32) ? 0xFF : 0x00;
    } else {
      add = 0;
    }
    for(j = 0; j < 5; j++) {
      LCD_data(ptr[j] | add);
    }
    LCD_data(0x00);
  }
  ++curBlink;
  for(i = 0; i < 16*6; ++i)
    LCD_data(0x08);
}

void HD44780_clear(void)
{
  register unsigned int n;
  register const char zero = ' ';
  for(n = 0; n < DDRAM_SIZE; n++)
    DDRAM[n] = zero;
}

void HD44780_cmd(char c)
{
  // TODO: dispatch HD44780 commands
  if(c & 0x80)        // DDRAM address !
  {
    // set cursor position
    // 00h-4Fh in 1-line mode
    // 00h-27h for first and 40-67h for second in 2-lines mode
    cursor = c & 0x7F;
    address = 0;
  }
  else if(c & 0x40)   // CGRAM address !
  {
    // set DDRAM address, next data puts to CGRAM instead of DDRAM
    address = c;
  }
  else if(c & 0x20)   // function
  {
    // data len (4-bit DL 8/4) lines (3-bit N 2/1) font (2-bit F 5x10/5x8)
    // (1,0-bits ignored)
    // cursor go to 2nd line after passes 40th pos of first line
    // scroll only horisontal, 2nd line never go to 1st line
  }
  else if(c & 0x10)   // shift
  {
    // move cursor and shift display without change DDRAM
    // (3-bit Display shift with cursor/Cursor move) (2-bit Right shift/Left shift)
    // (1,0-bits ignored)
  }
  else if(c & 0x08)   // display !
  {
    // display on/off (2-bit D) cursor on/off (1-bit C) blink on/off (0-bit B)
    if(c & 0x04)
      control |= DISPLAY_ON;
    else
      control &= ~DISPLAY_ON;
    if(c & 0x02)
      control |= CURSOR_ON;
    else
      control &= ~CURSOR_ON;
    if(c & 0x01)
      control |= CURSOR_BLINK;
    else
      control &= ~CURSOR_BLINK;
  }
  else if(c & 0x04)   // entry mode !
  {
    // move direction (1-bit I/D increment/decrement)
    // and display shift (0-bit S yes/no) depend on I/D left/right (only DDRAM not CGRAM)
  }
  else if(c & 0x02)   // home
  {
    // set pos to 0, undo all shifts (0-bit ignored)
    address = 0x80;
    shift = 0;
  }
  else if(c & 0x01)   // clear
  {
    // set 20H to all ddram (with fully blank), set pos to 0, undo all shifts
    // set increment mode (I/D = 1) and don't change S
    HD44780_clear();
    address = 0x80;
    shift = 0;
    control |= INCREMENT_MODE;
  }
  else                // nop
  {
    // nothing to do
  }
}

void HD44780_dat(char c)  // Output char with code (any)
{
  // TODO: dispatch HD44780 data (DDRAM or CGRAM write)
  if(address & 0x40)
  {
    // put cgram
    unsigned int ascii = (address & 0x3f) >> 3;
    unsigned char bit = 1 << (address & 0x07);
    unsigned char *ptr = &CGRAM[ascii * 5];
    if(c & 0x10)
      *ptr |= bit;
    else
      *ptr &= ~bit;
    ptr++;
    if(c & 0x08)
      *ptr |= bit;
    else
      *ptr &= ~bit;
    ptr++;
    if(c & 0x04)
      *ptr |= bit;
    else
      *ptr &= ~bit;
    ptr++;
    if(c & 0x02)
      *ptr |= bit;
    else
      *ptr &= ~bit;
    ptr++;
    if(c & 0x01)
      *ptr |= bit;
    else
      *ptr &= ~bit;
    address = 0x40 | ((address + 1) & 0x3F);
  }
  else
  {
    // put ddram
    DDRAM[cursor & 0x7F] = c;
    cursor = (cursor + 1) & 0x7F;
  }
}

// ---    UART ---

#pragma vector=UART1_R_RXNE_vector
__interrupt void UART1_R_RXNE_handler(void)
{
  if(!(UART1_SR & MASK_UART1_SR_RXNE) )
  {
    return;
  }
  char c = UART1_DR;
  // Implement used LCDTerm codes
  switch(state) {
  case 0:
    if(c == 0x03)
    {
      HD44780_cmd(1);
    }
    else if(c == 0x12)
    {
      state = 0x12;
    }
    else if(c == 0x14)
    {
      state = 0x14;
    }
    break;
    
  case 0x12:
    HD44780_cmd(c);
    state = 0;
    break;
    
  case 0x14:
    HD44780_dat(c);
    state = 0;
    break;
    
  default:
    state = 0;
  }
}


int main( void )
{
  CLK_CKDIVR  = 0;     //    (Fsys = Fosc = 16MHz)
  control = DISPLAY_ON | CURSOR_MOVE | CURSOR_ON | CURSOR_BLINK;
  address = 0x80;
  UART_init();
  // Run dispatcher as soon as possible
  __enable_interrupt();
  //  HSE_init();
  LCD_init();
  while(1) {
    HD44780_ScreenRedraw();
    _delay_ms(10);
  }
  __disable_interrupt();
  return 0;
}

