#include "iostm8.h"
#include "uart.h"
#include "intrinsics.h"
#include "lcd.h"

//  1, Flash 8, EEPROM 640 
//  UART1, SPI1, I2C1, TIM1, TIM2, TIM4, BEEP, AWU, WatchDog, ADC1

void _delay_ms(int ms) {
  char i, j;
  while(ms--)
    for(i = 12; i; --i)
      for(j = 255; j; --j)
        asm("nop");
}

void putString(const char* str)
{
  int len;
  for(len = 0; str[len]; ++len);
  UART_send(str, len);
}

const char hex[16] = {  '0', '1', '2', '3', '4', '5', '6', '7',
                        '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

void putHex(const char* data, int len)
{
  char c;
  while(len--) {
    c = *(data++);
    UART_send(&hex[(c>>4)&0x0F], 1);
    UART_send(&hex[c&0x0F], 1);
    c = ' ';
    UART_send(&c, 1);
  }
}

void putDump(const char* data, int len)
{
  char c;
  c = ' ';
  UART_send(&c, 1);
  while(len--) {
    c = *(data++);
    if(c < 32) c = '.';
    UART_send(&c, 1);
  }
  c = ' ';
  UART_send(&c, 1);
}

#define TERM_setReciver(func) UART_setRX(func)

#define Terminal  TerminalV52

#define TERM_WIDTH  16
#define TERM_HEIGHT 8
#define TERM_SIZE  (TERM_HEIGHT * TERM_WIDTH)
#define TERM_GETROW(pos)  ((pos)>>4)
#define TERM_GETCOL(pos)  ((pos)&0x0F)
#define TERM_GETLINE(pos) ((pos)&~0x0F)

#define MUTERM_ESC   27 

char termScreen[TERM_SIZE];  //   
unsigned char termAutoOverflow = 1;
unsigned char termPos = 0;
unsigned char termSavePos = 0;

#define TERMPOS_FIRST 0
#define TERMPOS_LAST  (TERM_HEIGHT * TERM_WIDTH - 1)


unsigned char curBlink = 0;

void ScreenRedraw() //  
{
  unsigned char scrOutPos;
  LCD_setpos(0, 0);
  for(scrOutPos = 0; scrOutPos < 128; ++scrOutPos) {
    if(scrOutPos == termPos && (curBlink & 32)) {
      LCD_data(0xFF);
      LCD_data(0xFF);
      LCD_data(0xFF);
      LCD_data(0xFF);
      LCD_data(0xFF);
      LCD_data(0xFF);
    } else {
      LCD_drawchar(termScreen[scrOutPos]);
    }
  }
  ++curBlink;
}


void TERM_clear()       //  
{
  register unsigned int n;
  register const char zero = 0;
  for(n = 0; n < TERM_SIZE; n++)
    termScreen[n] = zero;
}

void TERM_scrolldown()  //   (  )
{
  register unsigned int n = TERM_SIZE;
  register char* src = &termScreen[TERM_SIZE - TERM_WIDTH];
  while(n > TERM_WIDTH) {
    termScreen[--n] = *--src;
  }
  register const char zero = 0;
  do {
    termScreen[--n] = zero;
  } while(n);
}

void TERM_scrollup()    //   (  )
{
  register unsigned int n = 0;
  register char* src = &termScreen[TERM_WIDTH];
  while(n < TERM_SIZE - TERM_WIDTH) {
    termScreen[n++] = *src++;
  }
  register const char zero = 0;
  while(n < TERM_SIZE) {
    termScreen[n++] = zero;
  }
}

void TERM_clearRemaining()  //   
{
  register unsigned int n = termPos;
  register const char zero = 0;
  while(n < TERM_SIZE) {
    termScreen[n++] = zero;
  }
}

void TERM_clearFromStart()
{
  register unsigned int n = termPos;
  register const char zero = 0;
  while(n) {
    termScreen[n--] = zero;
  };
  termScreen[0] = zero;
}

void TERM_clearRestLine()   //   
{
  register unsigned int n = termPos;
  do {
    termScreen[n++] = 0;
  } while(TERM_GETCOL(n));
}

void TERM_clearStartLine()   //   
{
  register unsigned int n = termPos;
  while(TERM_GETCOL(n)) {
    termScreen[n--] = 0;
  }
  termScreen[n] = 0;
}

void TERM_deleteLine()      //     (74.5ms)
{
  register unsigned int n = (unsigned char)TERM_GETLINE(termPos);
  register char* src = &termScreen[n];
  while(n < TERM_SIZE - TERM_WIDTH) {
    termScreen[n++] = *src++;
  }
  register const char zero = 0;
  do {
    termScreen[n++] = zero;
  } while(n < TERM_SIZE);
}

void TERM_clearLine()       //  
{
  register unsigned int n = (unsigned char)TERM_GETLINE(termPos);
  do {
    termScreen[n++] = 0;
  } while(TERM_GETCOL(n));
}

void TERM_insertLine()      //      (97.2ms)
{
  register unsigned int idx = TERM_SIZE - TERM_WIDTH - 1;
  register unsigned int n = (unsigned char)TERM_GETLINE(TERM_SIZE - 1 - termPos);
  while(n) {
    register const char A = termScreen[idx];
    idx += TERM_WIDTH;
    termScreen[idx] = A;
    idx -= TERM_WIDTH + 1;
    n--;
  }
  idx += 1;
  do {
    termScreen[idx++] = 0;
  } while(TERM_GETCOL(idx));
}

void TERM_putch(char c) { // !!!
  termScreen[termPos] = c;
  if(TERM_GETCOL(termPos) != TERM_WIDTH - 1) {
    termPos++;
  } else if(termAutoOverflow) {
    termPos++;
    if(termPos == TERM_SIZE) {
      termPos = TERM_SIZE - TERM_WIDTH;
      TERM_scrollup();
    }
  }
}

void TerminalV52(char c);

void TerminalV52_ESC_DUMMY(char c)
{
  TERM_setReciver(TerminalV52);
}

void TerminalV52_ESC_Y2(char c)
{
  if( ((unsigned char)c) >= TERM_WIDTH ) c = TERM_WIDTH - 1;
  termPos |= c;
  TERM_setReciver(TerminalV52);
  //Note: The function @(Y,X) also returns such a control string.
}

void TerminalV52_ESC_Y(char c)
{
  if( ((unsigned char)c) >= TERM_HEIGHT) c = TERM_HEIGHT - 1;
  termPos = c * TERM_WIDTH;
  TERM_setReciver(TerminalV52_ESC_Y2);
}

void TerminalV52_ESC(char c)
{
    switch(c) {
    case 'A':   // UP
      if(termPos >= TERM_WIDTH) termPos -= TERM_WIDTH;
      break;
    case 'B':   // DOWN
      if(termPos < TERM_SIZE - TERM_WIDTH) termPos += TERM_WIDTH;
      break;
    case 'C':   // RIGHT
      if(TERM_GETCOL(termPos) < TERM_WIDTH - 1) ++termPos;
      break;
    case 'D':   // LEFT
      if(TERM_GETCOL(termPos)) --termPos;
      break;

    case 'E':   // Clear Home
      TERM_clear();
      termPos = 0;
      break;

    case 'H':   // Cursor Home
      termPos = 0;
      break;

    case 'I':   // Cursor Up
      if(termPos >= TERM_WIDTH)
        termPos -= TERM_WIDTH;
      else
        TERM_scrolldown();
      break;

    case 'J':   // Clear Remaining Screen
      TERM_clearRemaining();
      break;

    case 'K':   // Delete Rest of Line
      TERM_clearRestLine();
      break;

    case 'L':   // Insert Line
      TERM_insertLine();  //insert a blank line at the current cursor position.
      termPos = TERM_GETLINE(termPos);
      break;

    case 'M':   // Delete Line (with scroll rest)
      TERM_deleteLine();
      //termPos = TERM_SIZE - TERM_WIDTH; // first column of the added line?
      termPos = TERM_GETLINE(termPos);
      break;

    case 'Y':   // Positioning Cursor
      TERM_setReciver(TerminalV52_ESC_Y);
      //Note: The function @(Y,X) also returns such a control string.
      break;

    case 'b':   // Select Font Color
      TERM_setReciver(TerminalV52_ESC_DUMMY);
      //c = getchar();
      //set_fontcolor(c == '1' ? black : white); // 0 = white and 1 = black.
      //set_fontcolor(c&16);
      break;

    case 'c':   // Select Background Color
      TERM_setReciver(TerminalV52_ESC_DUMMY);
      //c = getchar();
      //set_groundcolor(c == '1' ? black : white); // 0 = white and 1 = black.
      //set_groundcolor(c&16);
      break;

    case 'd':   // Clear Screen up to Cursor Position
      TERM_clearFromStart();
      break;

    case 'e':   // Cursor On
      //cursor_visible(1);
      break;

    case 'f':   // Cursor Off
      //cursor_visible(0);
      break;

    case 'j':   // Save Cursor Position
      termSavePos = termPos;
      break;

    case 'k':   // Restore Cursor to Saved Position
      termPos = termSavePos;
      termSavePos = 0;  // for next restore to home if no save
      break;
      
    case 'l':   // Delete Line (without scroll rest)
      TERM_clearLine();
      termPos = TERM_GETLINE(termPos);
      break;

    case 'o':   // Delete Beginning of Line
      TERM_clearStartLine();
      break;

    case 'p':   // Reverse on
      //set_invert(1);
      break;

    case 'q':   // Reverse off
      //set_invert(0);
      break;

    case 'v':   // Automatic Overflow On
      termAutoOverflow = 1;
      break;

    case 'w':   // Automatic Overflow off
      termAutoOverflow = 0;
      break;
      
    case MUTERM_ESC:  // 
      break;
      
    default:    // Unknown Escape
      TERM_putch(MUTERM_ESC);
      TERM_putch(c);
      break;
    }
    TERM_setReciver(TerminalV52);
}

void TerminalV52(char c)
{
  if( ((unsigned char)c) >= 32 ) {
    TERM_putch(c);  //    
  } else
  switch(c) {
  case 7:       // BEL, Audio Output
    //beep();
    break;

  case 8:       // BS, Backspace
    if(TERM_GETCOL(termPos)) --termPos;
    termScreen[termPos] = 0; //below the cursor
    break;

  case 9:       // TAB, Tabulator
    c = termPos + 4;
    if(TERM_GETLINE(termPos) != TERM_GETLINE(c)) termPos |= 0x0F;
    else termPos = c & ~0x03;
    break;

  case 10:      // LF, Line Feed
    if(termPos < TERM_SIZE - TERM_WIDTH) termPos += TERM_WIDTH;
    else TERM_scrollup();
    //termPos = TERM_GETLINE(termPos);  //  
    break;

  case 12:      // FF, Form Feed
    //Effects closure of the current printer page. Most printers, however, do
    //not yet show any activity at this point. Output does not occur until the
    //entire document is closed using 'CHR$(26)'.
    break;

  case 13:      // CR, Carriage Return
    termPos = TERM_GETLINE(termPos);
    break;

  case 26:      // DE, Document End
    //Causes closure of the current printer page and output of the entire
    //document.
    break;
    
  case MUTERM_ESC:  // ESC-
    TERM_setReciver(TerminalV52_ESC);
    break;
  
  default:
    TERM_putch(c);
    break;
  }
}

void HSE_init()
{
  //    ( )
  CLK_ECKR_bit.HSEEN  = 1;      //  HSE
  while(!CLK_ECKR_bit.HSERDY);  //    
  CLK_SWCR_bit.SWEN   = 1;      //     
  CLK_SWR     = 0xB4;           //  HSE   
  CLK_CKDIVR  = 0;              //   
  while(!CLK_SWCR_bit.SWIF);    //   
  CLK_SWCR_bit.SWIF   = 0;
}

void init()
{
  HSE_init();
  UART_init();
  UART_setRX(TerminalV52);
  
  LCD_init();
  LCD_clear();
  putString("\r\nHello World\r\n");
  putString("My UID is:\r\n");
  putString("|  X  |  Y  |WN|     Lot Number     ||    DUMP    |\r\n ");
  putHex((const char*)0x4865, 12);
  putDump((const char*)0x4865, 12);
  putString("\r\nDUMP 0x4800 to 0x48FF:\r\n");
  const char* ptr;
  for(ptr = (const char*)0x4800; ptr <= (const char*)0x48FF; ptr += 16) {
    putString("0x0048");
    putHex(((const char*)&ptr)+1, 1);
    putHex(ptr, 16);
    putDump(ptr, 16);
    putString("\r\n");
  }
  putString("Init complite\r\n");
  UART_flush();
}

int main( void )
{
  init();
  //test();
  putString("Echo:\r\n");
  LCD_setpos(0,0);
  LCD_clear();
  __enable_interrupt();
  while(1) {
    ScreenRedraw();
    _delay_ms(10);
    //__wait_for_interrupt();
    /*for(c = '0'; c <= '9'; c++) {
      UART_send(&c, 1);
      _delay_ms(150);
    }
    putString("\r\n");
    _delay_ms(1000);*/
  }
  __disable_interrupt();
  return 0;
}

